<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  reports.php — Financial Reports & Analytics
 * ═══════════════════════════════════════════════════════════════
 *
 *  Reports included:
 *    1. Profit & Loss Summary (revenue - expenses = net)
 *    2. Income vs Expenses Chart (monthly bar chart)
 *    3. Top Clients by Revenue
 *    4. Expense Breakdown by Category (doughnut chart)
 *    5. Accounts Receivable Aging (0-30, 31-60, 61-90, 90+ days)
 *
 *  Features:
 *    - Configurable date range (presets + custom)
 *    - Printable layout
 *    - Chart.js visualizations
 *
 *  Authorization: Accountant only
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

/* ══════════════════════════════════════════════════════════════
 *  DATE RANGE HANDLING
 * ══════════════════════════════════════════════════════════════ */
$preset    = $_GET['preset'] ?? 'this_month';
$date_from = trim($_GET['date_from'] ?? '');
$date_to   = trim($_GET['date_to'] ?? '');

/* Preset ranges */
$today = date('Y-m-d');
$presets = [
    'this_month'    => [date('Y-m-01'), date('Y-m-t')],
    'last_month'    => [date('Y-m-01', strtotime('-1 month')), date('Y-m-t', strtotime('-1 month'))],
    'this_quarter'  => [date('Y-m-01', strtotime('first day of ' . ceil(date('n')/3)*3-2 . ' months ago January')), $today],
    'this_year'     => [date('Y-01-01'), date('Y-12-31')],
    'last_year'     => [date('Y-01-01', strtotime('-1 year')), date('Y-12-31', strtotime('-1 year'))],
    'all_time'      => ['2000-01-01', '2099-12-31'],
];

/* Calculate quarter start properly */
$quarter = ceil(date('n') / 3);
$quarter_start_month = ($quarter - 1) * 3 + 1;
$presets['this_quarter'] = [
    date('Y-m-01', strtotime(date('Y') . '-' . str_pad($quarter_start_month, 2, '0', STR_PAD_LEFT) . '-01')),
    $today
];

if ($preset === 'custom' && $date_from && $date_to) {
    $range_start = $date_from;
    $range_end   = $date_to;
} elseif (isset($presets[$preset])) {
    [$range_start, $range_end] = $presets[$preset];
} else {
    [$range_start, $range_end] = $presets['this_month'];
    $preset = 'this_month';
}

/* Human-readable range label */
$range_label = date(DATE_FORMAT, strtotime($range_start)) . ' – ' . date(DATE_FORMAT, strtotime($range_end));

/* ══════════════════════════════════════════════════════════════
 *  DATA QUERIES
 * ══════════════════════════════════════════════════════════════ */

/* ── 1. Profit & Loss ─────────────────────────────────────────── */
$invoice_revenue = (float) db_scalar(
    "SELECT COALESCE(SUM(amount_paid), 0) FROM invoices
     WHERE status != 'cancelled' AND issue_date BETWEEN ? AND ?",
    ['ss', $range_start, $range_end]
);

$other_income = (float) db_scalar(
    "SELECT COALESCE(SUM(amount), 0) FROM income
     WHERE income_date BETWEEN ? AND ?",
    ['ss', $range_start, $range_end]
);

$total_revenue = $invoice_revenue + $other_income;

$total_expenses = (float) db_scalar(
    "SELECT COALESCE(SUM(amount), 0) FROM expenses
     WHERE expense_date BETWEEN ? AND ?",
    ['ss', $range_start, $range_end]
);

$net_profit = $total_revenue - $total_expenses;

/* ── 2. Monthly Income vs Expenses (last 6 months for chart) ──── */
$monthly_data = [];
for ($i = 5; $i >= 0; $i--) {
    $m_start = date('Y-m-01', strtotime("-{$i} months"));
    $m_end   = date('Y-m-t', strtotime("-{$i} months"));
    $m_label = date('M Y', strtotime($m_start));

    $m_inv = (float) db_scalar(
        "SELECT COALESCE(SUM(amount_paid), 0) FROM invoices
         WHERE status != 'cancelled' AND issue_date BETWEEN ? AND ?",
        ['ss', $m_start, $m_end]
    );
    $m_inc = (float) db_scalar(
        "SELECT COALESCE(SUM(amount), 0) FROM income WHERE income_date BETWEEN ? AND ?",
        ['ss', $m_start, $m_end]
    );
    $m_exp = (float) db_scalar(
        "SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE expense_date BETWEEN ? AND ?",
        ['ss', $m_start, $m_end]
    );

    $monthly_data[] = [
        'label'   => $m_label,
        'income'  => $m_inv + $m_inc,
        'expense' => $m_exp,
    ];
}

/* ── 3. Top Clients by Revenue (within date range) ────────────── */
$top_clients = db_select(
    "SELECT c.id, c.company_name,
            COALESCE(SUM(i.amount_paid), 0) AS revenue,
            COUNT(DISTINCT i.id) AS invoice_count
     FROM clients c
     LEFT JOIN invoices i ON i.client_id = c.id
        AND i.status != 'cancelled'
        AND i.issue_date BETWEEN ? AND ?
     GROUP BY c.id, c.company_name
     HAVING revenue > 0
     ORDER BY revenue DESC
     LIMIT 10",
    ['ss', $range_start, $range_end]
);

/* ── 4. Expense Breakdown by Category ─────────────────────────── */
$expense_by_cat = db_select(
    "SELECT category, SUM(amount) AS total
     FROM expenses
     WHERE expense_date BETWEEN ? AND ?
     GROUP BY category
     ORDER BY total DESC",
    ['ss', $range_start, $range_end]
);

/* ── 5. Accounts Receivable Aging ─────────────────────────────── */
$aging_buckets = [
    '0-30'   => 0,
    '31-60'  => 0,
    '61-90'  => 0,
    '90+'    => 0,
];

$open_invoices = db_select(
    "SELECT id, invoice_number, client_id, due_date, balance_due,
            (SELECT company_name FROM clients WHERE id = invoices.client_id) AS company_name
     FROM invoices
     WHERE status NOT IN ('paid','cancelled') AND balance_due > 0"
);

foreach ($open_invoices as &$inv) {
    $days = (int) round((time() - strtotime($inv['due_date'])) / 86400);
    $inv['days_overdue'] = $days;

    if ($days <= 30)      $aging_buckets['0-30']  += (float)$inv['balance_due'];
    elseif ($days <= 60)  $aging_buckets['31-60'] += (float)$inv['balance_due'];
    elseif ($days <= 90)  $aging_buckets['61-90'] += (float)$inv['balance_due'];
    else                  $aging_buckets['90+']   += (float)$inv['balance_due'];
}
unset($inv);

$total_receivable = array_sum($aging_buckets);

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Financial Reports';
require_once __DIR__ . '/header.php';
?>

<style>
@media print {
    .lp-no-print { display: none !important; }
    .lp-card { break-inside: avoid; box-shadow: none !important; border: 1px solid #ddd !important; }
}
</style>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header lp-no-print">
    <div>
        <h1><i class="bi bi-bar-chart-line me-2" style="font-size:1.4rem"></i>Financial Reports</h1>
        <div class="lp-subtitle"><?php echo $range_label; ?></div>
    </div>
    <button class="btn btn-lp-outline lp-print-btn">
        <i class="bi bi-printer me-1"></i> Print Report
    </button>
</div>

<!-- ═══ Date Range Filter ═══════════════════════════════════════ -->
<div class="lp-card mb-4 lp-no-print">
    <div class="card-body py-3">
        <form method="GET" action="reports.php" class="row g-2 align-items-end">
            <div class="col-lg-3 col-md-4">
                <label class="form-label mb-1" style="font-size:.78rem; font-weight:600">Period</label>
                <select name="preset" id="presetSelect" class="form-select" style="font-size:.875rem">
                    <option value="this_month"   <?php echo $preset === 'this_month'   ? 'selected' : ''; ?>>This Month</option>
                    <option value="last_month"   <?php echo $preset === 'last_month'   ? 'selected' : ''; ?>>Last Month</option>
                    <option value="this_quarter" <?php echo $preset === 'this_quarter' ? 'selected' : ''; ?>>This Quarter</option>
                    <option value="this_year"    <?php echo $preset === 'this_year'    ? 'selected' : ''; ?>>This Year</option>
                    <option value="last_year"    <?php echo $preset === 'last_year'    ? 'selected' : ''; ?>>Last Year</option>
                    <option value="all_time"     <?php echo $preset === 'all_time'     ? 'selected' : ''; ?>>All Time</option>
                    <option value="custom"       <?php echo $preset === 'custom'       ? 'selected' : ''; ?>>Custom Range</option>
                </select>
            </div>
            <div class="col-lg-2 col-md-3 col-6" id="customFromWrap" style="<?php echo $preset !== 'custom' ? 'display:none' : ''; ?>">
                <label class="form-label mb-1" style="font-size:.78rem; font-weight:600">From</label>
                <input type="date" name="date_from" class="form-control" style="font-size:.835rem"
                       value="<?php echo h($date_from ?: $range_start); ?>">
            </div>
            <div class="col-lg-2 col-md-3 col-6" id="customToWrap" style="<?php echo $preset !== 'custom' ? 'display:none' : ''; ?>">
                <label class="form-label mb-1" style="font-size:.78rem; font-weight:600">To</label>
                <input type="date" name="date_to" class="form-control" style="font-size:.835rem"
                       value="<?php echo h($date_to ?: $range_end); ?>">
            </div>
            <div class="col-auto">
                <button type="submit" class="btn btn-lp-primary">
                    <i class="bi bi-arrow-repeat me-1"></i> Update
                </button>
            </div>
        </form>
    </div>
</div>

<script>
document.getElementById('presetSelect').addEventListener('change', function() {
    var show = this.value === 'custom';
    document.getElementById('customFromWrap').style.display = show ? '' : 'none';
    document.getElementById('customToWrap').style.display   = show ? '' : 'none';
});
</script>

<!-- ═══ Profit & Loss Summary ═══════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-header"><i class="bi bi-clipboard-data me-1"></i> Profit & Loss Summary</div>
    <div class="card-body">
        <div class="row g-3">
            <div class="col-md-4">
                <div class="p-3 rounded" style="background:var(--lp-emerald-soft)">
                    <div class="text-muted mb-1" style="font-size:.72rem; text-transform:uppercase; letter-spacing:.05em">Total Revenue</div>
                    <div class="fw-bold" style="font-size:1.5rem; color:var(--lp-emerald)"><?php echo format_currency($total_revenue); ?></div>
                    <div class="mt-2" style="font-size:.78rem; color:var(--lp-slate-mid)">
                        Invoice Payments: <?php echo format_currency($invoice_revenue); ?><br>
                        Other Income: <?php echo format_currency($other_income); ?>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="p-3 rounded" style="background:var(--lp-red-soft, #fef2f2)">
                    <div class="text-muted mb-1" style="font-size:.72rem; text-transform:uppercase; letter-spacing:.05em">Total Expenses</div>
                    <div class="fw-bold" style="font-size:1.5rem; color:var(--lp-red)"><?php echo format_currency($total_expenses); ?></div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="p-3 rounded" style="background:<?php echo $net_profit >= 0 ? 'var(--lp-blue-soft)' : 'var(--lp-amber-soft)'; ?>">
                    <div class="text-muted mb-1" style="font-size:.72rem; text-transform:uppercase; letter-spacing:.05em">Net Profit</div>
                    <div class="fw-bold" style="font-size:1.5rem; color:<?php echo $net_profit >= 0 ? 'var(--lp-blue)' : 'var(--lp-amber)'; ?>">
                        <?php echo format_currency($net_profit); ?>
                    </div>
                    <?php if ($total_revenue > 0): ?>
                    <div class="mt-2" style="font-size:.78rem; color:var(--lp-slate-mid)">
                        Margin: <?php echo number_format(($net_profit / $total_revenue) * 100, 1); ?>%
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- ═══ Charts Row ══════════════════════════════════════════════ -->
<div class="row g-4 mb-4">

    <!-- Monthly Income vs Expenses -->
    <div class="col-lg-8">
        <div class="lp-card h-100">
            <div class="card-header"><i class="bi bi-bar-chart me-1"></i> Income vs Expenses (Last 6 Months)</div>
            <div class="card-body">
                <canvas id="incomeExpenseChart" height="260"></canvas>
            </div>
        </div>
    </div>

    <!-- Expense by Category -->
    <div class="col-lg-4">
        <div class="lp-card h-100">
            <div class="card-header"><i class="bi bi-pie-chart me-1"></i> Expenses by Category</div>
            <div class="card-body d-flex align-items-center justify-content-center">
                <?php if (empty($expense_by_cat)): ?>
                    <div class="text-muted text-center py-4">No expenses in this period.</div>
                <?php else: ?>
                    <canvas id="expenseCatChart" height="240"></canvas>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- ═══ Top Clients & Aging Row ═════════════════════════════════ -->
<div class="row g-4 mb-4">

    <!-- Top Clients -->
    <div class="col-lg-6">
        <div class="lp-card h-100">
            <div class="card-header"><i class="bi bi-trophy me-1"></i> Top Clients by Revenue</div>
            <div class="card-body p-0">
                <?php if (empty($top_clients)): ?>
                    <div class="lp-empty py-4"><p>No client revenue in this period.</p></div>
                <?php else: ?>
                <div class="lp-table-responsive">
                    <table class="lp-table mb-0">
                        <thead>
                            <tr><th>Client</th><th class="text-center">Invoices</th><th class="text-end">Revenue</th></tr>
                        </thead>
                        <tbody>
                            <?php foreach ($top_clients as $tc): ?>
                            <tr>
                                <td>
                                    <a href="client_view.php?id=<?php echo (int)$tc['id']; ?>"
                                       class="text-decoration-none fw-semibold" style="color:var(--lp-navy)">
                                        <?php echo h($tc['company_name']); ?>
                                    </a>
                                </td>
                                <td class="text-center"><?php echo (int)$tc['invoice_count']; ?></td>
                                <td class="text-end fw-semibold" style="color:var(--lp-emerald)">
                                    <?php echo format_currency((float)$tc['revenue']); ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Accounts Receivable Aging -->
    <div class="col-lg-6">
        <div class="lp-card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
                <span><i class="bi bi-clock-history me-1"></i> Accounts Receivable Aging</span>
                <span class="fw-semibold" style="font-size:.85rem; color:var(--lp-amber)">
                    Total: <?php echo format_currency($total_receivable); ?>
                </span>
            </div>
            <div class="card-body">
                <div class="row g-3 mb-3">
                    <?php
                    $bucket_colors = ['0-30' => 'emerald', '31-60' => 'blue', '61-90' => 'amber', '90+' => 'red'];
                    foreach ($aging_buckets as $bucket => $amt):
                        $color = $bucket_colors[$bucket];
                    ?>
                    <div class="col-6 col-md-3">
                        <div class="p-2 rounded text-center" style="background:var(--lp-<?php echo $color; ?>-soft)">
                            <div class="text-muted" style="font-size:.68rem; text-transform:uppercase"><?php echo $bucket; ?> days</div>
                            <div class="fw-bold" style="font-size:1rem; color:var(--lp-<?php echo $color; ?>)">
                                <?php echo format_currency($amt); ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <?php if (!empty($open_invoices)): ?>
                <div class="lp-table-responsive" style="max-height:200px; overflow-y:auto">
                    <table class="lp-table mb-0" style="font-size:.82rem">
                        <thead style="position:sticky; top:0; background:var(--lp-white)">
                            <tr><th>Invoice</th><th>Client</th><th class="text-center">Days</th><th class="text-end">Balance</th></tr>
                        </thead>
                        <tbody>
                            <?php
                            usort($open_invoices, fn($a,$b) => $b['days_overdue'] <=> $a['days_overdue']);
                            foreach (array_slice($open_invoices, 0, 10) as $oi):
                            ?>
                            <tr>
                                <td>
                                    <a href="invoice_view.php?id=<?php echo (int)$oi['id']; ?>"
                                       class="text-decoration-none" style="color:var(--lp-navy)">
                                        <?php echo h($oi['invoice_number']); ?>
                                    </a>
                                </td>
                                <td><?php echo h($oi['company_name']); ?></td>
                                <td class="text-center">
                                    <span class="<?php echo $oi['days_overdue'] > 30 ? 'text-danger fw-semibold' : ''; ?>">
                                        <?php echo $oi['days_overdue']; ?>
                                    </span>
                                </td>
                                <td class="text-end fw-semibold" style="color:var(--lp-amber)">
                                    <?php echo format_currency((float)$oi['balance_due']); ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                    <div class="text-center text-muted py-3">No outstanding receivables.</div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- ═══ Chart.js ════════════════════════════════════════════════ -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.7/dist/chart.umd.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    /* Income vs Expense Bar Chart */
    var monthlyData = <?php echo json_encode($monthly_data); ?>;
    new Chart(document.getElementById('incomeExpenseChart'), {
        type: 'bar',
        data: {
            labels: monthlyData.map(d => d.label),
            datasets: [
                {
                    label: 'Income',
                    data: monthlyData.map(d => d.income),
                    backgroundColor: 'rgba(16, 185, 129, 0.7)',
                    borderColor: 'rgb(16, 185, 129)',
                    borderWidth: 1,
                    borderRadius: 4
                },
                {
                    label: 'Expenses',
                    data: monthlyData.map(d => d.expense),
                    backgroundColor: 'rgba(239, 68, 68, 0.7)',
                    borderColor: 'rgb(239, 68, 68)',
                    borderWidth: 1,
                    borderRadius: 4
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: { legend: { position: 'top' } },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: { callback: v => '<?php echo CURRENCY_SYMBOL; ?>' + v.toLocaleString() }
                }
            }
        }
    });

    <?php if (!empty($expense_by_cat)): ?>
    /* Expense by Category Doughnut */
    var catData = <?php echo json_encode($expense_by_cat); ?>;
    var catColors = [
        'rgb(239, 68, 68)', 'rgb(249, 115, 22)', 'rgb(234, 179, 8)',
        'rgb(34, 197, 94)', 'rgb(6, 182, 212)', 'rgb(59, 130, 246)',
        'rgb(139, 92, 246)', 'rgb(236, 72, 153)', 'rgb(107, 114, 128)'
    ];
    new Chart(document.getElementById('expenseCatChart'), {
        type: 'doughnut',
        data: {
            labels: catData.map(d => d.category),
            datasets: [{
                data: catData.map(d => parseFloat(d.total)),
                backgroundColor: catColors.slice(0, catData.length),
                borderWidth: 2,
                borderColor: '#fff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'bottom', labels: { boxWidth: 12, font: { size: 11 } } }
            }
        }
    });
    <?php endif; ?>
});
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
