<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  profile.php — User Profile & Password Change
 * ═══════════════════════════════════════════════════════════════
 *
 *  Features:
 *    - View profile information
 *    - Update name, email, phone
 *    - Change password (requires current password)
 *    - Avatar display (initials-based)
 *    - Last login timestamp
 *
 *  Authorization: Any logged-in user (their own profile only)
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_login();

$user_id = (int) $_SESSION['user_id'];
$errors  = [];
$success = '';

/* ── Load current user ────────────────────────────────────────── */
$user = db_select_one("SELECT * FROM users WHERE id = ?", ['i', $user_id]);
if (!$user) {
    flash('error', 'User not found.');
    redirect('dashboard.php');
}

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Profile Update or Password Change
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        $errors[] = 'Invalid security token.';
    }

    $action = $_POST['action'] ?? '';

    /* ── PROFILE UPDATE ────────────────────────────────────── */
    if ($action === 'update_profile' && empty($errors)) {
        $full_name = trim($_POST['full_name'] ?? '');
        $email     = trim($_POST['email'] ?? '');
        $phone     = trim($_POST['phone'] ?? '');

        if ($full_name === '') {
            $errors[] = 'Full name is required.';
        }
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'A valid email address is required.';
        } else {
            /* Check for duplicate email (exclude self) */
            $dup = db_select_one(
                "SELECT id FROM users WHERE email = ? AND id != ?",
                ['si', $email, $user_id]
            );
            if ($dup) {
                $errors[] = 'This email is already in use by another account.';
            }
        }

        if (empty($errors)) {
            db_execute(
                "UPDATE users SET full_name = ?, email = ?, phone = ? WHERE id = ?",
                ['sssi', $full_name, $email, $phone, $user_id]
            );

            /* Update session */
            $_SESSION['user_name']  = $full_name;
            $_SESSION['user_email'] = $email;

            /* Reload user data */
            $user = db_select_one("SELECT * FROM users WHERE id = ?", ['i', $user_id]);

            csrf_regenerate();
            $success = 'Profile updated successfully.';
        }
    }

    /* ── PASSWORD CHANGE ───────────────────────────────────── */
    if ($action === 'change_password' && empty($errors)) {
        $current_pass = $_POST['current_password'] ?? '';
        $new_pass     = $_POST['new_password'] ?? '';
        $confirm_pass = $_POST['confirm_password'] ?? '';

        if ($current_pass === '') {
            $errors[] = 'Current password is required.';
        } elseif (!password_verify($current_pass, $user['password_hash'])) {
            $errors[] = 'Current password is incorrect.';
        }

        if ($new_pass === '') {
            $errors[] = 'New password is required.';
        } elseif (strlen($new_pass) < PASSWORD_MIN_LEN) {
            $errors[] = 'New password must be at least ' . PASSWORD_MIN_LEN . ' characters.';
        }

        if ($new_pass !== $confirm_pass) {
            $errors[] = 'New passwords do not match.';
        }

        if (empty($errors)) {
            $new_hash = password_hash($new_pass, PASSWORD_BCRYPT);
            db_execute(
                "UPDATE users SET password_hash = ? WHERE id = ?",
                ['si', $new_hash, $user_id]
            );

            csrf_regenerate();
            $success = 'Password changed successfully.';
        }
    }
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'My Profile';
require_once __DIR__ . '/header.php';

/* Get initials for avatar */
$initials = '';
$name_parts = explode(' ', $user['full_name']);
foreach ($name_parts as $part) {
    $initials .= strtoupper(substr($part, 0, 1));
}
$initials = substr($initials, 0, 2);
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1><i class="bi bi-person-circle me-2" style="font-size:1.4rem"></i>My Profile</h1>
        <div class="lp-subtitle">Manage your account settings and password</div>
    </div>
    <a href="<?php echo APP_URL; ?>/dashboard.php" class="btn btn-lp-outline">
        <i class="bi bi-arrow-left me-1"></i> Dashboard
    </a>
</div>

<!-- ═══ Messages ════════════════════════════════════════════════ -->
<?php if (!empty($errors)): ?>
<div class="lp-flash error" style="animation:none;">
    <div>
        <i class="bi bi-exclamation-triangle-fill me-1"></i>
        <strong>Error:</strong>
        <?php echo h(implode(' ', $errors)); ?>
    </div>
</div>
<?php endif; ?>

<?php if ($success): ?>
<div class="lp-flash success" style="animation:none;">
    <div>
        <i class="bi bi-check-circle-fill me-1"></i>
        <?php echo h($success); ?>
    </div>
</div>
<?php endif; ?>

<div class="row g-4">

    <!-- ═══ Left: Profile Card ══════════════════════════════════ -->
    <div class="col-lg-4">
        <div class="lp-card">
            <div class="card-body text-center py-4">
                <!-- Avatar -->
                <div class="mx-auto mb-3" style="width:90px; height:90px; border-radius:50%;
                     background:linear-gradient(135deg, var(--lp-navy), var(--lp-blue));
                     display:flex; align-items:center; justify-content:center;
                     color:var(--lp-white); font-size:2rem; font-weight:700;
                     font-family:var(--lp-font-display); box-shadow:var(--lp-shadow-md)">
                    <?php echo h($initials); ?>
                </div>

                <!-- Name & Role -->
                <h4 class="mb-1" style="color:var(--lp-navy); font-family:var(--lp-font-display)">
                    <?php echo h($user['full_name']); ?>
                </h4>
                <span class="badge rounded-pill mb-3"
                      style="background:<?php echo $user['role'] === ROLE_ACCOUNTANT ? 'var(--lp-emerald-soft)' : 'var(--lp-blue-soft)'; ?>;
                             color:<?php echo $user['role'] === ROLE_ACCOUNTANT ? 'var(--lp-emerald)' : 'var(--lp-blue)'; ?>;
                             font-size:.72rem; font-weight:600; padding:.4em .8em">
                    <?php echo ucfirst(h($user['role'])); ?>
                </span>

                <!-- Info list -->
                <div class="text-start mt-3 pt-3" style="border-top:1px solid var(--lp-silver)">
                    <div class="d-flex align-items-center mb-2" style="font-size:.85rem">
                        <i class="bi bi-envelope text-muted me-2" style="width:20px"></i>
                        <span><?php echo h($user['email']); ?></span>
                    </div>
                    <?php if ($user['phone']): ?>
                    <div class="d-flex align-items-center mb-2" style="font-size:.85rem">
                        <i class="bi bi-telephone text-muted me-2" style="width:20px"></i>
                        <span><?php echo h($user['phone']); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="d-flex align-items-center mb-2" style="font-size:.85rem">
                        <i class="bi bi-calendar3 text-muted me-2" style="width:20px"></i>
                        <span>Joined <?php echo date(DATE_FORMAT, strtotime($user['created_at'])); ?></span>
                    </div>
                    <?php if ($user['last_login_at']): ?>
                    <div class="d-flex align-items-center" style="font-size:.85rem">
                        <i class="bi bi-clock-history text-muted me-2" style="width:20px"></i>
                        <span>Last login <?php echo date(DATETIME_FORMAT, strtotime($user['last_login_at'])); ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Status -->
                <div class="mt-3 pt-3" style="border-top:1px solid var(--lp-silver)">
                    <span class="d-inline-flex align-items-center gap-1" style="font-size:.82rem">
                        <span style="width:8px; height:8px; border-radius:50%;
                              background:<?php echo $user['is_active'] ? 'var(--lp-emerald)' : 'var(--lp-red)'; ?>"></span>
                        <?php echo $user['is_active'] ? 'Active Account' : 'Inactive Account'; ?>
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- ═══ Right: Forms ════════════════════════════════════════ -->
    <div class="col-lg-8">

        <!-- Profile Form -->
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-person me-1"></i> Profile Information
            </div>
            <div class="card-body">
                <form method="POST" action="profile.php" class="lp-form" novalidate>
                    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                    <input type="hidden" name="action" value="update_profile">

                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Full Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="full_name"
                                   value="<?php echo h($user['full_name']); ?>"
                                   required maxlength="100">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Email Address <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="email"
                                   value="<?php echo h($user['email']); ?>"
                                   required maxlength="150">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Phone Number</label>
                            <input type="tel" class="form-control" name="phone"
                                   value="<?php echo h($user['phone'] ?? ''); ?>"
                                   maxlength="30" placeholder="Optional">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Role</label>
                            <input type="text" class="form-control" disabled
                                   value="<?php echo ucfirst(h($user['role'])); ?>"
                                   style="background:var(--lp-snow)">
                            <small class="text-muted" style="font-size:.72rem">Contact administrator to change role.</small>
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-lp-primary">
                            <i class="bi bi-check-lg me-1"></i> Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Password Change Form -->
        <div class="lp-card">
            <div class="card-header">
                <i class="bi bi-shield-lock me-1"></i> Change Password
            </div>
            <div class="card-body">
                <form method="POST" action="profile.php" class="lp-form" novalidate>
                    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                    <input type="hidden" name="action" value="change_password">

                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">Current Password <span class="text-danger">*</span></label>
                            <div class="position-relative">
                                <input type="password" class="form-control" name="current_password"
                                       id="currentPass" required autocomplete="current-password">
                                <button type="button" class="btn btn-sm position-absolute top-50 end-0 translate-middle-y me-2 p-0 border-0"
                                        onclick="togglePass('currentPass', this)" style="background:transparent">
                                    <i class="bi bi-eye text-muted"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">New Password <span class="text-danger">*</span></label>
                            <div class="position-relative">
                                <input type="password" class="form-control" name="new_password"
                                       id="newPass" required minlength="<?php echo PASSWORD_MIN_LEN; ?>"
                                       autocomplete="new-password">
                                <button type="button" class="btn btn-sm position-absolute top-50 end-0 translate-middle-y me-2 p-0 border-0"
                                        onclick="togglePass('newPass', this)" style="background:transparent">
                                    <i class="bi bi-eye text-muted"></i>
                                </button>
                            </div>
                            <small class="text-muted" style="font-size:.72rem">
                                Minimum <?php echo PASSWORD_MIN_LEN; ?> characters
                            </small>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Confirm New Password <span class="text-danger">*</span></label>
                            <div class="position-relative">
                                <input type="password" class="form-control" name="confirm_password"
                                       id="confirmPass" required autocomplete="new-password">
                                <button type="button" class="btn btn-sm position-absolute top-50 end-0 translate-middle-y me-2 p-0 border-0"
                                        onclick="togglePass('confirmPass', this)" style="background:transparent">
                                    <i class="bi bi-eye text-muted"></i>
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-lp-primary">
                            <i class="bi bi-shield-check me-1"></i> Update Password
                        </button>
                    </div>
                </form>
            </div>
        </div>

    </div>
</div>

<script>
function togglePass(inputId, btn) {
    var input = document.getElementById(inputId);
    var icon  = btn.querySelector('i');
    if (input.type === 'password') {
        input.type = 'text';
        icon.classList.remove('bi-eye');
        icon.classList.add('bi-eye-slash');
    } else {
        input.type = 'password';
        icon.classList.remove('bi-eye-slash');
        icon.classList.add('bi-eye');
    }
}
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
