<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  logout.php — Secure Session Teardown
 * ═══════════════════════════════════════════════════════════════
 *
 *  This file performs a complete, secure logout:
 *
 *    1. Starts the session (if not already active)
 *    2. Wipes all session variables from memory
 *    3. Destroys the session file on the server
 *    4. Expires the session cookie in the browser
 *    5. Sets a flash message for the login page
 *    6. Redirects to login.php
 *
 *  Why each step matters:
 *    - $_SESSION = [] clears data from the current request
 *    - session_destroy() removes the server-side session file
 *    - setcookie(..., time()-3600) tells the browser to delete
 *      the cookie immediately — prevents session reuse
 *    - Redirect prevents the user from pressing "back" into
 *      a logged-in state
 *
 *  Accessed via:
 *    GET /logout.php  (from the navbar "Sign Out" link)
 */

require_once __DIR__ . '/config.php';

/* ── 1. Ensure session is started ─────────────────────────────── */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/* ── 2. Capture the session name and cookie params ────────────── */
$session_name   = session_name();
$cookie_params  = session_get_cookie_params();

/* ── 3. Wipe all session variables ────────────────────────────── */
$_SESSION = [];

/* ── 4. Destroy the server-side session file ──────────────────── */
session_destroy();

/* ── 5. Expire the session cookie in the browser ──────────────── */
setcookie(
    $session_name,            // Cookie name (usually PHPSESSID)
    '',                       // Empty value
    time() - 3600,            // Expired 1 hour ago
    $cookie_params['path'],
    $cookie_params['domain'],
    $cookie_params['secure'],
    $cookie_params['httponly']
);

/* ── 6. Start a fresh session for the flash message ───────────── */
session_start();
flash('success', 'You have been signed out successfully.');

/* ── 7. Redirect to login ─────────────────────────────────────── */
redirect('login.php');
