<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  invoices.php — Invoice Management List
 * ═══════════════════════════════════════════════════════════════
 *
 *  Features:
 *    - Search by invoice number or client name
 *    - Filter by status (draft/sent/paid/overdue/cancelled)
 *    - Filter by specific client (accountant only)
 *    - Paginated results with preserved query params
 *    - Delete with CSRF + FK safety check
 *    - Desktop table + mobile card view
 *
 *  Authorization:
 *    - Accountant: sees all invoices
 *    - Client: sees only their own invoices (enforced in WHERE)
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_login();

/* ══════════════════════════════════════════════════════════════
 *  HANDLE DELETE (POST, accountant only)
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete' && is_accountant()) {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
        redirect('invoices.php');
    }

    $del_id = (int) ($_POST['invoice_id'] ?? 0);

    if ($del_id > 0) {
        /* Check for payments — RESTRICT if payments exist */
        $pay_count = (int) db_scalar(
            "SELECT COUNT(*) FROM payments WHERE invoice_id = ?",
            ['i', $del_id]
        );

        if ($pay_count > 0) {
            flash('error', "Cannot delete — this invoice has {$pay_count} payment(s) recorded. Cancel it instead.");
        } else {
            /* Line items cascade automatically via FK */
            db_execute("DELETE FROM invoices WHERE id = ?", ['i', $del_id]);
            flash('success', 'Invoice deleted successfully.');
        }
    }

    csrf_regenerate();
    redirect('invoices.php');
}

/* ══════════════════════════════════════════════════════════════
 *  FILTERS & SEARCH
 * ══════════════════════════════════════════════════════════════ */
$search    = trim($_GET['search'] ?? '');
$status    = $_GET['status']    ?? 'all';
$client_filter = (int) ($_GET['client_id'] ?? 0);
$date_from = trim($_GET['date_from'] ?? '');
$date_to   = trim($_GET['date_to']   ?? '');

/* Build WHERE dynamically */
$where  = [];
$params = [];
$types  = '';

/* Client role: lock to own client_id */
if (is_client()) {
    $cid      = (int) ($_SESSION['client_id'] ?? 0);
    $where[]  = "i.client_id = ?";
    $params[] = $cid;
    $types   .= 'i';
}

/* Search: invoice number or client name */
if ($search !== '') {
    $safe     = db_escape_like($search);
    $where[]  = "(i.invoice_number LIKE CONCAT('%',?,'%') OR c.company_name LIKE CONCAT('%',?,'%'))";
    $params[] = $safe;
    $params[] = $safe;
    $types   .= 'ss';
}

/* Status filter */
$valid_statuses = ['draft','sent','paid','overdue','cancelled'];
if ($status !== 'all' && in_array($status, $valid_statuses)) {
    if ($status === 'overdue') {
        /* Include dynamically overdue invoices */
        $where[] = "(i.status = 'overdue' OR (i.status IN ('sent','draft') AND i.due_date < CURDATE()))";
    } else {
        $where[]  = "i.status = ?";
        $params[] = $status;
        $types   .= 's';
    }
}

/* Client filter (accountant only) */
if ($client_filter > 0 && is_accountant()) {
    $where[]  = "i.client_id = ?";
    $params[] = $client_filter;
    $types   .= 'i';
}

/* Date range */
if ($date_from !== '') {
    $where[]  = "i.issue_date >= ?";
    $params[] = $date_from;
    $types   .= 's';
}
if ($date_to !== '') {
    $where[]  = "i.issue_date <= ?";
    $params[] = $date_to;
    $types   .= 's';
}

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

/* ── Pagination ───────────────────────────────────────────────── */
$count_params = empty($types) ? [] : array_merge([$types], $params);
$pager = db_paginate(
    "SELECT COUNT(*) FROM invoices i JOIN clients c ON c.id = i.client_id {$where_sql}",
    $count_params
);

/* ── Fetch invoices ───────────────────────────────────────────── */
$sel_types  = $types . 'ii';
$sel_params = array_merge($params, [$pager['limit'], $pager['offset']]);

$invoices = db_select(
    "SELECT i.*, c.company_name
     FROM invoices i
     JOIN clients c ON c.id = i.client_id
     {$where_sql}
     ORDER BY i.created_at DESC
     LIMIT ? OFFSET ?",
    array_merge([$sel_types], $sel_params)
);

/* ── Client list for filter dropdown (accountant only) ────────── */
$client_list = [];
if (is_accountant()) {
    $client_list = db_select(
        "SELECT id, company_name FROM clients WHERE is_active = 1 ORDER BY company_name ASC"
    );
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = is_accountant() ? 'Invoices' : 'My Invoices';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1>
            <i class="bi bi-receipt me-2" style="font-size:1.4rem"></i>
            <?php echo is_accountant() ? 'Invoices' : 'My Invoices'; ?>
        </h1>
        <div class="lp-subtitle">
            <?php echo $pager['total_rows']; ?> invoice<?php echo $pager['total_rows'] !== 1 ? 's' : ''; ?> found
        </div>
    </div>
    <?php if (is_accountant()): ?>
    <a href="<?php echo APP_URL; ?>/invoice_form.php" class="btn btn-lp-primary">
        <i class="bi bi-plus-lg me-1"></i> New Invoice
    </a>
    <?php endif; ?>
</div>

<!-- ═══ Search & Filter Bar ═════════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-body py-3">
        <form method="GET" action="invoices.php" class="row g-2 align-items-end">

            <!-- Search -->
            <div class="col-lg-3 col-md-6">
                <div class="input-group">
                    <span class="input-group-text bg-white border-end-0" style="border-color:var(--lp-silver)">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input type="text" name="search" class="form-control border-start-0 ps-0"
                           style="border-color:var(--lp-silver); font-size:.875rem"
                           placeholder="Invoice # or client…"
                           value="<?php echo h($search); ?>">
                </div>
            </div>

            <!-- Status -->
            <div class="col-lg-2 col-md-3 col-6">
                <select name="status" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all"       <?php echo $status === 'all'       ? 'selected' : ''; ?>>All Statuses</option>
                    <option value="draft"     <?php echo $status === 'draft'     ? 'selected' : ''; ?>>Draft</option>
                    <option value="sent"      <?php echo $status === 'sent'      ? 'selected' : ''; ?>>Sent</option>
                    <option value="paid"      <?php echo $status === 'paid'      ? 'selected' : ''; ?>>Paid</option>
                    <option value="overdue"   <?php echo $status === 'overdue'   ? 'selected' : ''; ?>>Overdue</option>
                    <option value="cancelled" <?php echo $status === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                </select>
            </div>

            <?php if (is_accountant()): ?>
            <!-- Client filter -->
            <div class="col-lg-2 col-md-3 col-6">
                <select name="client_id" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="0">All Clients</option>
                    <?php foreach ($client_list as $cl): ?>
                    <option value="<?php echo (int)$cl['id']; ?>"
                        <?php echo $client_filter === (int)$cl['id'] ? 'selected' : ''; ?>>
                        <?php echo h($cl['company_name']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>

            <!-- Date range -->
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_from" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_from); ?>"
                       title="From date">
            </div>
            <div class="col-lg-1 col-md-3 col-6">
                <input type="date" name="date_to" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_to); ?>"
                       title="To date">
            </div>

            <!-- Buttons -->
            <div class="col-lg-2 col-md-6 d-flex gap-2">
                <button type="submit" class="btn btn-lp-primary">
                    <i class="bi bi-funnel me-1"></i> Filter
                </button>
                <?php
                $has_filters = ($search !== '' || $status !== 'all' || $client_filter > 0 || $date_from !== '' || $date_to !== '');
                if ($has_filters):
                ?>
                <a href="<?php echo APP_URL; ?>/invoices.php" class="btn btn-lp-outline">
                    <i class="bi bi-x-lg me-1"></i> Clear
                </a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- ═══ Invoice List ════════════════════════════════════════════ -->
<?php if (empty($invoices)): ?>
    <div class="lp-card">
        <div class="lp-empty">
            <i class="bi bi-receipt d-block"></i>
            <p>
                <?php echo $has_filters
                    ? 'No invoices match your filters.'
                    : 'No invoices yet. Create your first invoice to get started.'; ?>
            </p>
            <?php if (!$has_filters && is_accountant()): ?>
            <a href="<?php echo APP_URL; ?>/invoice_form.php" class="btn btn-lp-primary mt-3">
                <i class="bi bi-plus-lg me-1"></i> New Invoice
            </a>
            <?php endif; ?>
        </div>
    </div>
<?php else: ?>

<!-- Desktop table -->
<div class="lp-card mb-4 d-none d-md-block">
    <div class="lp-table-responsive">
        <table class="lp-table">
            <thead>
                <tr>
                    <th>Invoice #</th>
                    <th>Client</th>
                    <th>Status</th>
                    <th>Issue Date</th>
                    <th>Due Date</th>
                    <th class="text-end">Total</th>
                    <th class="text-end">Paid</th>
                    <th class="text-end">Balance</th>
                    <?php if (is_accountant()): ?>
                    <th class="text-end">Actions</th>
                    <?php endif; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($invoices as $inv):
                    /* Compute effective status */
                    $eff = $inv['status'];
                    if (in_array($inv['status'], ['sent','draft']) && $inv['due_date'] < date('Y-m-d')) {
                        $eff = 'overdue';
                    }
                ?>
                <tr>
                    <td>
                        <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$inv['id']; ?>"
                           class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                            <?php echo h($inv['invoice_number']); ?>
                        </a>
                    </td>
                    <td>
                        <?php if (is_accountant()): ?>
                        <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$inv['client_id']; ?>"
                           class="text-decoration-none" style="color:var(--lp-blue); font-size:.84rem">
                            <?php echo h($inv['company_name']); ?>
                        </a>
                        <?php else: ?>
                            <?php echo h($inv['company_name']); ?>
                        <?php endif; ?>
                    </td>
                    <td><span class="lp-badge <?php echo h($eff); ?>"><?php echo h(ucfirst($eff)); ?></span></td>
                    <td style="font-size:.84rem"><?php echo date(DATE_FORMAT, strtotime($inv['issue_date'])); ?></td>
                    <td>
                        <span class="<?php echo $eff === 'overdue' ? 'text-danger fw-semibold' : ''; ?>"
                              style="font-size:.84rem">
                            <?php echo date(DATE_FORMAT, strtotime($inv['due_date'])); ?>
                        </span>
                        <?php if ($eff === 'overdue'):
                            $days_over = (int) round((time() - strtotime($inv['due_date'])) / 86400);
                        ?>
                        <small class="d-block text-danger" style="font-size:.68rem">
                            <?php echo $days_over; ?>d overdue
                        </small>
                        <?php endif; ?>
                    </td>
                    <td class="text-end"><?php echo format_currency((float)$inv['total']); ?></td>
                    <td class="text-end" style="color:var(--lp-emerald)">
                        <?php echo format_currency((float)$inv['amount_paid']); ?>
                    </td>
                    <td class="text-end fw-semibold">
                        <?php if ((float)$inv['balance_due'] > 0): ?>
                            <span style="color:var(--lp-amber)"><?php echo format_currency((float)$inv['balance_due']); ?></span>
                        <?php else: ?>
                            <span class="text-muted"><?php echo format_currency(0); ?></span>
                        <?php endif; ?>
                    </td>
                    <?php if (is_accountant()): ?>
                    <td class="text-end">
                        <div class="dropdown">
                            <button class="btn btn-sm btn-lp-outline dropdown-toggle" type="button"
                                    data-bs-toggle="dropdown" style="padding:.3rem .65rem; font-size:.78rem">
                                Actions
                            </button>
                            <ul class="dropdown-menu dropdown-menu-end shadow-sm">
                                <li>
                                    <a class="dropdown-item" href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$inv['id']; ?>">
                                        <i class="bi bi-eye me-2"></i> View
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="<?php echo APP_URL; ?>/invoice_form.php?id=<?php echo (int)$inv['id']; ?>">
                                        <i class="bi bi-pencil me-2"></i> Edit
                                    </a>
                                </li>
                                <?php if ($inv['status'] !== 'paid'): ?>
                                <li>
                                    <a class="dropdown-item" href="<?php echo APP_URL; ?>/payment_form.php?invoice_id=<?php echo (int)$inv['id']; ?>">
                                        <i class="bi bi-credit-card me-2"></i> Record Payment
                                    </a>
                                </li>
                                <?php endif; ?>
                                <li><hr class="dropdown-divider"></li>
                                <li>
                                    <form method="POST" action="invoices.php" class="px-3 py-1"
                                          data-confirm="Delete invoice <?php echo h($inv['invoice_number']); ?>? This cannot be undone.">
                                        <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                                               value="<?php echo csrf_token(); ?>">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="invoice_id" value="<?php echo (int)$inv['id']; ?>">
                                        <button type="submit" class="dropdown-item text-danger p-0">
                                            <i class="bi bi-trash me-2"></i> Delete
                                        </button>
                                    </form>
                                </li>
                            </ul>
                        </div>
                    </td>
                    <?php endif; ?>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Mobile cards -->
<div class="d-md-none mb-4">
    <?php foreach ($invoices as $inv):
        $eff = $inv['status'];
        if (in_array($inv['status'], ['sent','draft']) && $inv['due_date'] < date('Y-m-d')) {
            $eff = 'overdue';
        }
    ?>
    <div class="lp-card mb-3">
        <div class="card-body">
            <!-- Top: invoice # + status -->
            <div class="d-flex align-items-start justify-content-between mb-2">
                <div>
                    <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$inv['id']; ?>"
                       class="fw-semibold text-decoration-none d-block" style="color:var(--lp-navy); font-size:.95rem">
                        <?php echo h($inv['invoice_number']); ?>
                    </a>
                    <small class="text-muted"><?php echo h($inv['company_name']); ?></small>
                </div>
                <span class="lp-badge <?php echo h($eff); ?>"><?php echo h(ucfirst($eff)); ?></span>
            </div>

            <!-- Dates -->
            <div class="d-flex gap-3 mb-3" style="font-size:.82rem">
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Issued</small>
                    <?php echo date(DATE_FORMAT, strtotime($inv['issue_date'])); ?>
                </div>
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Due</small>
                    <span class="<?php echo $eff === 'overdue' ? 'text-danger fw-semibold' : ''; ?>">
                        <?php echo date(DATE_FORMAT, strtotime($inv['due_date'])); ?>
                    </span>
                </div>
            </div>

            <!-- Amounts -->
            <div class="d-flex gap-3 mb-3 pb-3" style="border-bottom:1px solid var(--lp-silver)">
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Total</small>
                    <span class="fw-semibold" style="color:var(--lp-navy)"><?php echo format_currency((float)$inv['total']); ?></span>
                </div>
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Paid</small>
                    <span style="color:var(--lp-emerald)"><?php echo format_currency((float)$inv['amount_paid']); ?></span>
                </div>
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Balance</small>
                    <span class="fw-semibold" style="color:<?php echo (float)$inv['balance_due'] > 0 ? 'var(--lp-amber)' : 'var(--lp-slate-mid)'; ?>">
                        <?php echo format_currency((float)$inv['balance_due']); ?>
                    </span>
                </div>
            </div>

            <!-- Actions -->
            <div class="d-flex gap-2 flex-wrap">
                <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$inv['id']; ?>"
                   class="btn btn-sm btn-lp-outline">
                    <i class="bi bi-eye me-1"></i> View
                </a>
                <?php if (is_accountant()): ?>
                <a href="<?php echo APP_URL; ?>/invoice_form.php?id=<?php echo (int)$inv['id']; ?>"
                   class="btn btn-sm btn-lp-outline">
                    <i class="bi bi-pencil me-1"></i> Edit
                </a>
                <?php if ($inv['status'] !== 'paid'): ?>
                <a href="<?php echo APP_URL; ?>/payment_form.php?invoice_id=<?php echo (int)$inv['id']; ?>"
                   class="btn btn-sm btn-lp-primary">
                    <i class="bi bi-credit-card me-1"></i> Pay
                </a>
                <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- ═══ Pagination ══════════════════════════════════════════════ -->
<?php if ($pager['total_pages'] > 1): ?>
<nav aria-label="Invoice pagination">
    <ul class="pagination lp-pagination justify-content-center mb-0">
        <li class="page-item <?php echo !$pager['has_prev'] ? 'disabled' : ''; ?>">
            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] - 1])); ?>">
                <i class="bi bi-chevron-left"></i>
            </a>
        </li>
        <?php
        $range = 2;
        $start = max(1, $pager['current'] - $range);
        $end   = min($pager['total_pages'], $pager['current'] + $range);

        if ($start > 1): ?>
            <li class="page-item">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => 1])); ?>">1</a>
            </li>
            <?php if ($start > 2): ?>
            <li class="page-item disabled"><span class="page-link">…</span></li>
            <?php endif;
        endif;

        for ($p = $start; $p <= $end; $p++): ?>
            <li class="page-item <?php echo $p === $pager['current'] ? 'active' : ''; ?>">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $p])); ?>">
                    <?php echo $p; ?>
                </a>
            </li>
        <?php endfor;

        if ($end < $pager['total_pages']): ?>
            <?php if ($end < $pager['total_pages'] - 1): ?>
            <li class="page-item disabled"><span class="page-link">…</span></li>
            <?php endif; ?>
            <li class="page-item">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['total_pages']])); ?>">
                    <?php echo $pager['total_pages']; ?>
                </a>
            </li>
        <?php endif; ?>

        <li class="page-item <?php echo !$pager['has_next'] ? 'disabled' : ''; ?>">
            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] + 1])); ?>">
                <i class="bi bi-chevron-right"></i>
            </a>
        </li>
    </ul>
</nav>
<?php endif; ?>

<?php endif; /* end if !empty($invoices) */ ?>

<?php require_once __DIR__ . '/footer.php'; ?>
