<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  invoice_view.php — Invoice Detail & Print View
 * ═══════════════════════════════════════════════════════════════
 *
 *  Displays:
 *    - Full invoice document (printable layout)
 *    - Line items table with totals
 *    - Payment history for this invoice
 *    - Status update form (accountant only)
 *    - Quick actions: edit, record payment, print, delete
 *
 *  Authorization:
 *    - Accountant: any invoice
 *    - Client: own invoices only (enforced via client_id check)
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_login();

/* ══════════════════════════════════════════════════════════════
 *  LOAD INVOICE
 * ══════════════════════════════════════════════════════════════ */
$invoice_id = isset($_GET['id']) ? (int) $_GET['id'] : 0;

if ($invoice_id <= 0) {
    flash('error', 'Invalid invoice.');
    redirect('invoices.php');
}

$invoice = db_select_one(
    "SELECT i.*, c.company_name, c.contact_name, c.email AS client_email,
            c.phone AS client_phone, c.tax_id,
            c.address_line1, c.address_line2, c.city, c.state,
            c.postal_code, c.country
     FROM invoices i
     JOIN clients c ON c.id = i.client_id
     WHERE i.id = ?",
    ['i', $invoice_id]
);

if (!$invoice) {
    flash('error', 'Invoice not found.');
    redirect('invoices.php');
}

/* Client role: verify ownership */
if (is_client()) {
    $my_client_id = (int) ($_SESSION['client_id'] ?? 0);
    if ((int)$invoice['client_id'] !== $my_client_id) {
        flash('error', 'Access denied.');
        redirect('dashboard.php');
    }
}

/* ══════════════════════════════════════════════════════════════
 *  HANDLE STATUS UPDATE (POST, accountant only)
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'update_status' && is_accountant()) {
    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
    } else {
        $new_status = $_POST['new_status'] ?? '';
        $valid = ['draft','sent','paid','overdue','cancelled'];
        if (in_array($new_status, $valid)) {
            db_execute(
                "UPDATE invoices SET status = ? WHERE id = ?",
                ['si', $new_status, $invoice_id]
            );
            csrf_regenerate();
            flash('success', 'Invoice status updated to ' . ucfirst($new_status) . '.');
            redirect('invoice_view.php?id=' . $invoice_id);
        }
    }
}

/* ── Line Items ───────────────────────────────────────────────── */
$items = db_select(
    "SELECT * FROM invoice_items WHERE invoice_id = ? ORDER BY sort_order ASC",
    ['i', $invoice_id]
);

/* ── Payments ─────────────────────────────────────────────────── */
$payments = db_select(
    "SELECT p.*, u.full_name AS recorder_name
     FROM payments p
     LEFT JOIN users u ON u.id = p.recorded_by
     WHERE p.invoice_id = ?
     ORDER BY p.payment_date DESC",
    ['i', $invoice_id]
);

/* ── Effective status ─────────────────────────────────────────── */
$eff_status = $invoice['status'];
if (in_array($invoice['status'], ['sent','draft']) && $invoice['due_date'] < date('Y-m-d')) {
    $eff_status = 'overdue';
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Invoice ' . $invoice['invoice_number'];
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Print Styles ════════════════════════════════════════════ -->
<style>
    @media print {
        .lp-navbar, .lp-footer, .lp-no-print,
        .lp-page-header, .lp-flash { display: none !important; }
        .lp-page { padding: 0 !important; }
        .lp-card { box-shadow: none !important; border: 1px solid #ddd !important; break-inside: avoid; }
        body { background: #fff !important; }
        .lp-invoice-doc { border: none !important; box-shadow: none !important; }
    }

    .lp-invoice-doc {
        background: var(--lp-white);
        border: 1px solid var(--lp-silver);
        border-radius: var(--lp-radius-lg);
        box-shadow: var(--lp-shadow-lg);
        overflow: hidden;
    }
    .lp-inv-header {
        background: var(--lp-navy);
        color: var(--lp-white);
        padding: 2rem 2.25rem;
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        flex-wrap: wrap;
        gap: 1.5rem;
    }
    .lp-inv-brand {
        font-family: var(--lp-font-display);
        font-size: 1.75rem;
        display: flex;
        align-items: center;
        gap: .6rem;
    }
    .lp-inv-brand-icon {
        width: 40px; height: 40px;
        background: rgba(255,255,255,.12);
        border-radius: var(--lp-radius);
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.1rem;
    }
    .lp-inv-title {
        text-align: right;
    }
    .lp-inv-title h2 {
        font-family: var(--lp-font-display);
        font-size: 1.5rem;
        color: var(--lp-white);
        margin: 0;
    }
    .lp-inv-title .inv-number {
        font-family: var(--lp-font-body);
        font-size: .85rem;
        opacity: .7;
        margin-top: .25rem;
    }
    .lp-inv-body { padding: 2rem 2.25rem; }

    .lp-inv-addresses {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
        margin-bottom: 2rem;
    }
    @media (max-width: 576px) {
        .lp-inv-addresses { grid-template-columns: 1fr; gap: 1.25rem; }
        .lp-inv-header { padding: 1.5rem; }
        .lp-inv-body { padding: 1.5rem; }
    }
    .lp-inv-addr-label {
        font-size: .7rem;
        text-transform: uppercase;
        letter-spacing: .06em;
        font-weight: 700;
        color: var(--lp-slate-mid);
        margin-bottom: .4rem;
    }
    .lp-inv-addr {
        font-size: .88rem;
        line-height: 1.65;
        color: var(--lp-slate);
    }
    .lp-inv-addr strong {
        color: var(--lp-navy);
        font-size: .92rem;
    }

    .lp-inv-meta {
        display: flex;
        flex-wrap: wrap;
        gap: 1.5rem;
        margin-bottom: 2rem;
        padding-bottom: 1.5rem;
        border-bottom: 1px solid var(--lp-silver);
    }
    .lp-inv-meta-item {
        min-width: 120px;
    }
    .lp-inv-meta-label {
        font-size: .7rem;
        text-transform: uppercase;
        letter-spacing: .06em;
        font-weight: 700;
        color: var(--lp-slate-mid);
        margin-bottom: .2rem;
    }
    .lp-inv-meta-value {
        font-size: .9rem;
        font-weight: 600;
        color: var(--lp-navy);
    }
</style>

<!-- ═══ Page Header (not printed) ══════════════════════════════ -->
<div class="lp-page-header lp-no-print">
    <div>
        <h1>
            <i class="bi bi-receipt me-2" style="font-size:1.4rem"></i>
            Invoice <?php echo h($invoice['invoice_number']); ?>
        </h1>
        <div class="lp-subtitle d-flex align-items-center gap-2 flex-wrap">
            <span class="lp-badge <?php echo h($eff_status); ?>"><?php echo ucfirst(h($eff_status)); ?></span>
            <span><?php echo h($invoice['company_name']); ?></span>
            <span class="text-muted">&middot;</span>
            <span class="text-muted">Created <?php echo date(DATE_FORMAT, strtotime($invoice['created_at'])); ?></span>
        </div>
    </div>
    <div class="d-flex gap-2 flex-wrap">
        <button class="btn btn-lp-outline lp-print-btn">
            <i class="bi bi-printer me-1"></i> Print
        </button>
        <?php if (is_accountant()): ?>
        <a href="<?php echo APP_URL; ?>/invoice_form.php?id=<?php echo $invoice_id; ?>"
           class="btn btn-lp-outline">
            <i class="bi bi-pencil me-1"></i> Edit
        </a>
        <?php if ($invoice['status'] !== 'paid'): ?>
        <a href="<?php echo APP_URL; ?>/payment_form.php?invoice_id=<?php echo $invoice_id; ?>"
           class="btn btn-lp-primary">
            <i class="bi bi-credit-card me-1"></i> Record Payment
        </a>
        <?php endif; ?>
        <?php endif; ?>
        <a href="<?php echo APP_URL; ?>/invoices.php" class="btn btn-lp-outline">
            <i class="bi bi-arrow-left me-1"></i> Back
        </a>
    </div>
</div>

<div class="row g-4">

    <!-- ═══ Left: Invoice Document ══════════════════════════════ -->
    <div class="col-lg-8">
        <div class="lp-invoice-doc mb-4">

            <!-- Header bar -->
            <div class="lp-inv-header">
                <div class="lp-inv-brand">
                    <span class="lp-inv-brand-icon"><i class="bi bi-journal-bookmark-fill"></i></span>
                    <?php echo APP_NAME; ?>
                </div>
                <div class="lp-inv-title">
                    <h2>INVOICE</h2>
                    <div class="inv-number"><?php echo h($invoice['invoice_number']); ?></div>
                </div>
            </div>

            <!-- Body -->
            <div class="lp-inv-body">

                <!-- Addresses -->
                <div class="lp-inv-addresses">
                    <div>
                        <div class="lp-inv-addr-label">From</div>
                        <div class="lp-inv-addr">
                            <strong><?php echo APP_NAME; ?></strong><br>
                            Your Business Address<br>
                            City, State ZIP<br>
                            info@ledgerpro.com
                        </div>
                    </div>
                    <div>
                        <div class="lp-inv-addr-label">Bill To</div>
                        <div class="lp-inv-addr">
                            <strong><?php echo h($invoice['company_name']); ?></strong><br>
                            <?php echo h($invoice['contact_name']); ?><br>
                            <?php if ($invoice['address_line1']): ?>
                                <?php echo h($invoice['address_line1']); ?><br>
                            <?php endif; ?>
                            <?php if ($invoice['address_line2']): ?>
                                <?php echo h($invoice['address_line2']); ?><br>
                            <?php endif; ?>
                            <?php
                            $city_parts = array_filter([
                                $invoice['city'], $invoice['state'], $invoice['postal_code']
                            ]);
                            if (!empty($city_parts)):
                            ?>
                                <?php echo h(implode(', ', $city_parts)); ?><br>
                            <?php endif; ?>
                            <?php echo h($invoice['client_email']); ?>
                            <?php if ($invoice['tax_id']): ?>
                                <br><small class="text-muted">Tax ID: <?php echo h($invoice['tax_id']); ?></small>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Meta strip -->
                <div class="lp-inv-meta">
                    <div class="lp-inv-meta-item">
                        <div class="lp-inv-meta-label">Invoice Date</div>
                        <div class="lp-inv-meta-value"><?php echo date(DATE_FORMAT, strtotime($invoice['issue_date'])); ?></div>
                    </div>
                    <div class="lp-inv-meta-item">
                        <div class="lp-inv-meta-label">Due Date</div>
                        <div class="lp-inv-meta-value <?php echo $eff_status === 'overdue' ? 'text-danger' : ''; ?>">
                            <?php echo date(DATE_FORMAT, strtotime($invoice['due_date'])); ?>
                            <?php if ($eff_status === 'overdue'):
                                $days_over = (int) round((time() - strtotime($invoice['due_date'])) / 86400);
                            ?>
                                <small class="d-block text-danger" style="font-size:.72rem; font-weight:400">
                                    <?php echo $days_over; ?> day<?php echo $days_over !== 1 ? 's' : ''; ?> overdue
                                </small>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="lp-inv-meta-item">
                        <div class="lp-inv-meta-label">Status</div>
                        <div class="lp-inv-meta-value">
                            <span class="lp-badge <?php echo h($eff_status); ?>"><?php echo ucfirst(h($eff_status)); ?></span>
                        </div>
                    </div>
                    <div class="lp-inv-meta-item">
                        <div class="lp-inv-meta-label">Balance Due</div>
                        <div class="lp-inv-meta-value" style="color:<?php echo (float)$invoice['balance_due'] > 0 ? 'var(--lp-amber)' : 'var(--lp-emerald)'; ?>">
                            <?php echo format_currency((float)$invoice['balance_due']); ?>
                        </div>
                    </div>
                </div>

                <!-- Line Items Table -->
                <div class="lp-table-responsive mb-3">
                    <table class="lp-table">
                        <thead>
                            <tr>
                                <th style="width:5%">#</th>
                                <th style="width:50%">Description</th>
                                <th class="text-center" style="width:12%">Qty</th>
                                <th class="text-end" style="width:16%">Unit Price</th>
                                <th class="text-end" style="width:17%">Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($items as $idx => $itm): ?>
                            <tr>
                                <td class="text-muted"><?php echo $idx + 1; ?></td>
                                <td><?php echo h($itm['description']); ?></td>
                                <td class="text-center"><?php echo rtrim(rtrim(number_format((float)$itm['quantity'], 2), '0'), '.'); ?></td>
                                <td class="text-end"><?php echo format_currency((float)$itm['unit_price']); ?></td>
                                <td class="text-end fw-semibold"><?php echo format_currency((float)$itm['line_total']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Totals -->
                <div class="row justify-content-end">
                    <div class="col-sm-7 col-md-6 col-lg-5">
                        <div class="d-flex justify-content-between mb-2" style="font-size:.9rem">
                            <span class="text-muted">Subtotal</span>
                            <span class="fw-semibold"><?php echo format_currency((float)$invoice['subtotal']); ?></span>
                        </div>
                        <?php if ((float)$invoice['tax_rate'] > 0): ?>
                        <div class="d-flex justify-content-between mb-2" style="font-size:.9rem">
                            <span class="text-muted">Tax (<?php echo rtrim(rtrim(number_format((float)$invoice['tax_rate'], 2), '0'), '.'); ?>%)</span>
                            <span><?php echo format_currency((float)$invoice['tax_amount']); ?></span>
                        </div>
                        <?php endif; ?>
                        <?php if ((float)$invoice['discount_amount'] > 0): ?>
                        <div class="d-flex justify-content-between mb-2" style="font-size:.9rem">
                            <span class="text-muted">Discount</span>
                            <span class="text-danger">-<?php echo format_currency((float)$invoice['discount_amount']); ?></span>
                        </div>
                        <?php endif; ?>

                        <div class="d-flex justify-content-between py-2 mb-2"
                             style="border-top:2px solid var(--lp-navy); border-bottom:1px solid var(--lp-silver); font-size:1.05rem">
                            <span class="fw-bold" style="color:var(--lp-navy)">Total</span>
                            <span class="fw-bold" style="color:var(--lp-navy)"><?php echo format_currency((float)$invoice['total']); ?></span>
                        </div>

                        <?php if ((float)$invoice['amount_paid'] > 0): ?>
                        <div class="d-flex justify-content-between mb-2" style="font-size:.9rem">
                            <span class="text-muted">Paid</span>
                            <span style="color:var(--lp-emerald)">-<?php echo format_currency((float)$invoice['amount_paid']); ?></span>
                        </div>
                        <?php endif; ?>

                        <div class="d-flex justify-content-between pt-1" style="font-size:1.1rem">
                            <span class="fw-bold" style="color:<?php echo (float)$invoice['balance_due'] > 0 ? 'var(--lp-amber)' : 'var(--lp-emerald)'; ?>">
                                Balance Due
                            </span>
                            <span class="fw-bold" style="color:<?php echo (float)$invoice['balance_due'] > 0 ? 'var(--lp-amber)' : 'var(--lp-emerald)'; ?>">
                                <?php echo format_currency((float)$invoice['balance_due']); ?>
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Notes & Terms -->
                <?php if ($invoice['notes'] || $invoice['terms']): ?>
                <div class="mt-4 pt-3" style="border-top:1px solid var(--lp-silver)">
                    <div class="row g-3">
                        <?php if ($invoice['notes']): ?>
                        <div class="col-md-6">
                            <div class="lp-inv-addr-label">Notes</div>
                            <div style="font-size:.84rem; color:var(--lp-slate-mid); line-height:1.6; white-space:pre-wrap"><?php echo h($invoice['notes']); ?></div>
                        </div>
                        <?php endif; ?>
                        <?php if ($invoice['terms']): ?>
                        <div class="col-md-6">
                            <div class="lp-inv-addr-label">Terms & Conditions</div>
                            <div style="font-size:.84rem; color:var(--lp-slate-mid); line-height:1.6; white-space:pre-wrap"><?php echo h($invoice['terms']); ?></div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

            </div><!-- /inv-body -->
        </div><!-- /invoice-doc -->
    </div><!-- /left col -->

    <!-- ═══ Right: Sidebar ══════════════════════════════════════ -->
    <div class="col-lg-4 lp-no-print">

        <!-- Status Management (accountant only) -->
        <?php if (is_accountant()): ?>
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-arrow-repeat me-1"></i> Update Status
            </div>
            <div class="card-body">
                <form method="POST" action="invoice_view.php?id=<?php echo $invoice_id; ?>">
                    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                           value="<?php echo csrf_token(); ?>">
                    <input type="hidden" name="action" value="update_status">
                    <div class="d-flex gap-2">
                        <select name="new_status" class="form-select form-select-sm"
                                style="font-size:.84rem">
                            <?php
                            $statuses = ['draft'=>'Draft','sent'=>'Sent','paid'=>'Paid','overdue'=>'Overdue','cancelled'=>'Cancelled'];
                            foreach ($statuses as $val => $label):
                            ?>
                            <option value="<?php echo $val; ?>"
                                <?php echo $invoice['status'] === $val ? 'selected' : ''; ?>>
                                <?php echo $label; ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <button type="submit" class="btn btn-sm btn-lp-primary" style="white-space:nowrap">
                            Update
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>

        <!-- Payment History -->
        <div class="lp-card mb-4">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-credit-card me-1"></i> Payments</span>
                <span class="fw-semibold" style="font-size:.82rem; color:var(--lp-emerald)">
                    <?php echo format_currency((float)$invoice['amount_paid']); ?> paid
                </span>
            </div>
            <div class="card-body p-0">
                <?php if (empty($payments)): ?>
                    <div class="lp-empty" style="padding:2rem 1rem">
                        <i class="bi bi-credit-card d-block"></i>
                        <p>No payments recorded yet.</p>
                    </div>
                <?php else: ?>
                <div class="list-group list-group-flush">
                    <?php foreach ($payments as $pay): ?>
                    <div class="list-group-item border-0 px-3 py-2" style="border-bottom:1px solid #f1f5f9 !important">
                        <div class="d-flex align-items-center justify-content-between">
                            <div>
                                <div class="fw-semibold" style="font-size:.84rem; color:var(--lp-navy)">
                                    <?php echo date(DATE_FORMAT, strtotime($pay['payment_date'])); ?>
                                </div>
                                <small class="text-muted">
                                    <?php echo h(ucfirst(str_replace('_', ' ', $pay['method']))); ?>
                                    <?php if ($pay['reference']): ?>
                                        &middot; <?php echo h($pay['reference']); ?>
                                    <?php endif; ?>
                                </small>
                                <?php if ($pay['recorder_name']): ?>
                                <small class="d-block text-muted" style="font-size:.72rem">
                                    by <?php echo h($pay['recorder_name']); ?>
                                </small>
                                <?php endif; ?>
                            </div>
                            <span class="fw-semibold" style="color:var(--lp-emerald); font-size:.9rem">
                                +<?php echo format_currency((float)$pay['amount']); ?>
                            </span>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>

                <?php if (is_accountant() && $invoice['status'] !== 'paid'): ?>
                <div class="p-3 border-top" style="border-color:var(--lp-silver) !important">
                    <a href="<?php echo APP_URL; ?>/payment_form.php?invoice_id=<?php echo $invoice_id; ?>"
                       class="btn btn-sm btn-lp-primary w-100">
                        <i class="bi bi-plus-lg me-1"></i> Record Payment
                    </a>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Quick Info -->
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-info-circle me-1"></i> Details
            </div>
            <div class="card-body" style="font-size:.84rem">
                <div class="d-flex justify-content-between mb-2">
                    <span class="text-muted">Invoice ID</span>
                    <span class="fw-semibold">#<?php echo $invoice_id; ?></span>
                </div>
                <div class="d-flex justify-content-between mb-2">
                    <span class="text-muted">Client</span>
                    <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$invoice['client_id']; ?>"
                       class="text-decoration-none fw-semibold" style="color:var(--lp-blue)">
                        <?php echo h($invoice['company_name']); ?>
                    </a>
                </div>
                <div class="d-flex justify-content-between mb-2">
                    <span class="text-muted">Line Items</span>
                    <span><?php echo count($items); ?></span>
                </div>
                <div class="d-flex justify-content-between mb-2">
                    <span class="text-muted">Created</span>
                    <span><?php echo date(DATETIME_FORMAT, strtotime($invoice['created_at'])); ?></span>
                </div>
                <div class="d-flex justify-content-between">
                    <span class="text-muted">Last Updated</span>
                    <span><?php echo date(DATETIME_FORMAT, strtotime($invoice['updated_at'])); ?></span>
                </div>
            </div>
        </div>

        <!-- Actions -->
        <?php if (is_accountant()): ?>
        <div class="lp-card">
            <div class="card-body d-grid gap-2">
                <a href="<?php echo APP_URL; ?>/invoice_form.php?id=<?php echo $invoice_id; ?>"
                   class="btn btn-lp-outline">
                    <i class="bi bi-pencil me-1"></i> Edit Invoice
                </a>
                <form method="POST" action="invoices.php"
                      data-confirm="Delete invoice <?php echo h($invoice['invoice_number']); ?>? This cannot be undone.">
                    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                           value="<?php echo csrf_token(); ?>">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="invoice_id" value="<?php echo $invoice_id; ?>">
                    <button type="submit" class="btn btn-lp-danger w-100">
                        <i class="bi bi-trash me-1"></i> Delete Invoice
                    </button>
                </form>
            </div>
        </div>
        <?php endif; ?>

    </div><!-- /sidebar -->
</div><!-- /row -->

<?php require_once __DIR__ . '/footer.php'; ?>
