<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  income.php — Non-Invoice Income Tracking
 * ═══════════════════════════════════════════════════════════════
 *
 *  Tracks revenue that doesn't come through invoices:
 *    interest, refunds, grants, consulting, misc income, etc.
 *
 *  Features:
 *    - Stat cards: total income, this month, entry count
 *    - Search by description or category
 *    - Filter by category, client, date range
 *    - Create / Edit / Delete with CSRF
 *    - Responsive table + mobile cards
 *
 *  Authorization: Accountant only
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

/* ── Income categories ────────────────────────────────────────── */
$categories = [
    'Consulting', 'Interest', 'Refund', 'Grant', 'Commission',
    'Royalty', 'Rental', 'Investment', 'Miscellaneous',
];

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Create / Edit / Delete
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
        redirect('income.php');
    }

    $action = $_POST['action'] ?? '';

    /* ── DELETE ─────────────────────────────────────────────── */
    if ($action === 'delete') {
        $del_id = (int) ($_POST['income_id'] ?? 0);
        if ($del_id > 0) {
            db_execute("DELETE FROM income WHERE id = ?", ['i', $del_id]);
            flash('success', 'Income entry deleted.');
        }
        csrf_regenerate();
        redirect('income.php');
    }

    /* ── CREATE / UPDATE ───────────────────────────────────── */
    if ($action === 'save') {
        $edit_id     = (int) ($_POST['edit_id'] ?? 0);
        $client_id   = (int) ($_POST['client_id'] ?? 0);
        $category    = trim($_POST['category'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $amount      = trim($_POST['amount'] ?? '');
        $income_date = trim($_POST['income_date'] ?? '');
        $reference   = trim($_POST['reference'] ?? '');
        $notes       = trim($_POST['notes'] ?? '');

        $errors = [];
        if ($category === '')     $errors[] = 'Category is required.';
        if ($description === '')  $errors[] = 'Description is required.';
        if (!is_numeric($amount) || (float)$amount <= 0) $errors[] = 'A valid positive amount is required.';
        if ($income_date === '' || !strtotime($income_date)) $errors[] = 'A valid date is required.';

        if (!empty($errors)) {
            flash('error', implode(' ', $errors));
        } else {
            $client_val = $client_id > 0 ? $client_id : null;

            if ($edit_id > 0) {
                db_execute(
                    "UPDATE income SET client_id = ?, category = ?, description = ?,
                        amount = ?, income_date = ?, reference = ?, notes = ?
                     WHERE id = ?",
                    ['issdsssi',
                     $client_val, $category, $description,
                     (float)$amount, $income_date, $reference, $notes,
                     $edit_id]
                );
                flash('success', 'Income entry updated.');
            } else {
                db_insert(
                    "INSERT INTO income
                        (client_id, category, description, amount, income_date, reference, notes, recorded_by)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                    ['issdssi',  // Note: nullable int passed as 'i' or null
                     $client_val, $category, $description,
                     (float)$amount, $income_date, $reference, $notes,
                     $_SESSION['user_id']]
                );
                flash('success', 'Income entry created.');
            }
        }
        csrf_regenerate();
        redirect('income.php');
    }
}

/* ══════════════════════════════════════════════════════════════
 *  FILTERS
 * ══════════════════════════════════════════════════════════════ */
$search    = trim($_GET['search'] ?? '');
$cat_filter = trim($_GET['category'] ?? 'all');
$date_from = trim($_GET['date_from'] ?? '');
$date_to   = trim($_GET['date_to'] ?? '');

$where  = [];
$params = [];
$types  = '';

if ($search !== '') {
    $safe = db_escape_like($search);
    $where[]  = "(n.description LIKE CONCAT('%',?,'%') OR n.reference LIKE CONCAT('%',?,'%'))";
    $params[] = $safe;
    $params[] = $safe;
    $types   .= 'ss';
}
if ($cat_filter !== 'all' && $cat_filter !== '') {
    $where[]  = "n.category = ?";
    $params[] = $cat_filter;
    $types   .= 's';
}
if ($date_from !== '') {
    $where[] = "n.income_date >= ?"; $params[] = $date_from; $types .= 's';
}
if ($date_to !== '') {
    $where[] = "n.income_date <= ?"; $params[] = $date_to; $types .= 's';
}

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

/* ── Stats ────────────────────────────────────────────────────── */
$sp = empty($types) ? [] : array_merge([$types], $params);
$stat_total = (float) db_scalar("SELECT COALESCE(SUM(n.amount),0) FROM income n {$where_sql}", $sp);
$stat_count = (int)   db_scalar("SELECT COUNT(*) FROM income n {$where_sql}", $sp);
$stat_month = (float) db_scalar(
    "SELECT COALESCE(SUM(amount),0) FROM income WHERE income_date >= ?",
    ['s', date('Y-m-01')]
);

/* ── Pagination ───────────────────────────────────────────────── */
$pager = db_paginate("SELECT COUNT(*) FROM income n {$where_sql}", $sp);

$sel_types  = $types . 'ii';
$sel_params = array_merge($params, [$pager['limit'], $pager['offset']]);

$rows = db_select(
    "SELECT n.*, c.company_name
     FROM income n
     LEFT JOIN clients c ON c.id = n.client_id
     {$where_sql}
     ORDER BY n.income_date DESC, n.id DESC
     LIMIT ? OFFSET ?",
    array_merge([$sel_types], $sel_params)
);

/* ── Client list for form ─────────────────────────────────────── */
$client_list = db_select("SELECT id, company_name FROM clients WHERE is_active = 1 ORDER BY company_name ASC");

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Income';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1><i class="bi bi-graph-up-arrow me-2" style="font-size:1.4rem"></i>Income</h1>
        <div class="lp-subtitle">Non-invoice revenue and miscellaneous income</div>
    </div>
    <button class="btn btn-lp-primary" data-bs-toggle="modal" data-bs-target="#incomeModal"
            onclick="resetIncomeForm()">
        <i class="bi bi-plus-lg me-1"></i> Add Income
    </button>
</div>

<!-- ═══ Stats ═══════════════════════════════════════════════════ -->
<div class="row g-3 mb-4">
    <div class="col-sm-4">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon emerald"><i class="bi bi-cash-stack"></i></div>
            <div><div class="lp-stat-label">Total Income</div>
                 <div class="lp-stat-value"><?php echo format_currency($stat_total); ?></div></div>
        </div></div>
    </div>
    <div class="col-sm-4">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon blue"><i class="bi bi-calendar-check"></i></div>
            <div><div class="lp-stat-label">This Month</div>
                 <div class="lp-stat-value"><?php echo format_currency($stat_month); ?></div></div>
        </div></div>
    </div>
    <div class="col-sm-4">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon purple"><i class="bi bi-hash"></i></div>
            <div><div class="lp-stat-label">Entries</div>
                 <div class="lp-stat-value"><?php echo $stat_count; ?></div></div>
        </div></div>
    </div>
</div>

<!-- ═══ Filters ═════════════════════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-body py-3">
        <form method="GET" action="income.php" class="row g-2 align-items-end">
            <div class="col-lg-3 col-md-6">
                <div class="input-group">
                    <span class="input-group-text bg-white border-end-0" style="border-color:var(--lp-silver)">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input type="text" name="search" class="form-control border-start-0 ps-0"
                           style="border-color:var(--lp-silver); font-size:.875rem"
                           placeholder="Search description…"
                           value="<?php echo h($search); ?>">
                </div>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <select name="category" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all">All Categories</option>
                    <?php foreach ($categories as $cat): ?>
                    <option value="<?php echo h($cat); ?>" <?php echo $cat_filter === $cat ? 'selected' : ''; ?>>
                        <?php echo h($cat); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_from" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_from); ?>" title="From">
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_to" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_to); ?>" title="To">
            </div>
            <div class="col-lg-3 col-md-3 col-6 d-flex gap-2">
                <button type="submit" class="btn btn-lp-primary"><i class="bi bi-funnel me-1"></i> Filter</button>
                <?php $hf = ($search !== '' || $cat_filter !== 'all' || $date_from !== '' || $date_to !== '');
                if ($hf): ?>
                <a href="income.php" class="btn btn-lp-outline"><i class="bi bi-x-lg me-1"></i> Clear</a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- ═══ Income List ═════════════════════════════════════════════ -->
<?php if (empty($rows)): ?>
    <div class="lp-card"><div class="lp-empty">
        <i class="bi bi-graph-up-arrow d-block"></i>
        <p><?php echo $hf ? 'No entries match your filters.' : 'No income entries yet.'; ?></p>
    </div></div>
<?php else: ?>

<!-- Desktop -->
<div class="lp-card mb-4 d-none d-md-block">
    <div class="lp-table-responsive">
        <table class="lp-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Category</th>
                    <th>Description</th>
                    <th>Client</th>
                    <th>Reference</th>
                    <th class="text-end">Amount</th>
                    <th class="text-end">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($rows as $r): ?>
                <tr>
                    <td style="font-size:.84rem"><?php echo date(DATE_FORMAT, strtotime($r['income_date'])); ?></td>
                    <td>
                        <span class="badge rounded-pill"
                              style="background:var(--lp-emerald-soft); color:var(--lp-emerald); font-size:.72rem; font-weight:500">
                            <?php echo h($r['category']); ?>
                        </span>
                    </td>
                    <td><?php echo h($r['description']); ?></td>
                    <td style="font-size:.84rem; color:var(--lp-slate-mid)">
                        <?php echo h($r['company_name'] ?? '—'); ?>
                    </td>
                    <td style="font-size:.84rem; color:var(--lp-slate-mid)">
                        <?php echo h($r['reference'] ?: '—'); ?>
                    </td>
                    <td class="text-end fw-semibold" style="color:var(--lp-emerald)">
                        +<?php echo format_currency((float)$r['amount']); ?>
                    </td>
                    <td class="text-end">
                        <button class="btn btn-sm btn-lp-outline me-1" style="padding:.2rem .5rem; font-size:.78rem"
                                onclick="editIncome(<?php echo h(json_encode($r)); ?>)">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <form method="POST" action="income.php" class="d-inline"
                              data-confirm="Delete this income entry?">
                            <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="income_id" value="<?php echo (int)$r['id']; ?>">
                            <button type="submit" class="btn btn-sm text-danger border-0"
                                    style="padding:.2rem .5rem; font-size:.8rem"><i class="bi bi-trash"></i></button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Mobile -->
<div class="d-md-none mb-4">
    <?php foreach ($rows as $r): ?>
    <div class="lp-card mb-3"><div class="card-body">
        <div class="d-flex justify-content-between align-items-start mb-2">
            <div>
                <div class="fw-semibold" style="color:var(--lp-navy); font-size:.92rem"><?php echo h($r['description']); ?></div>
                <small class="text-muted"><?php echo h($r['company_name'] ?? 'No client'); ?></small>
            </div>
            <span class="fw-bold" style="color:var(--lp-emerald)">+<?php echo format_currency((float)$r['amount']); ?></span>
        </div>
        <div class="d-flex gap-3 mb-2" style="font-size:.8rem">
            <span><i class="bi bi-calendar3 me-1 text-muted"></i><?php echo date(DATE_FORMAT, strtotime($r['income_date'])); ?></span>
            <span class="badge rounded-pill" style="background:var(--lp-emerald-soft); color:var(--lp-emerald); font-size:.7rem"><?php echo h($r['category']); ?></span>
        </div>
        <div class="d-flex gap-2 pt-2" style="border-top:1px solid var(--lp-silver)">
            <button class="btn btn-sm btn-lp-outline" style="font-size:.78rem"
                    onclick="editIncome(<?php echo h(json_encode($r)); ?>)">
                <i class="bi bi-pencil me-1"></i> Edit
            </button>
            <form method="POST" action="income.php" class="d-inline" data-confirm="Delete?">
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="income_id" value="<?php echo (int)$r['id']; ?>">
                <button type="submit" class="btn btn-sm text-danger border-0" style="font-size:.78rem">
                    <i class="bi bi-trash me-1"></i> Delete
                </button>
            </form>
        </div>
    </div></div>
    <?php endforeach; ?>
</div>

<!-- Pagination -->
<?php if ($pager['total_pages'] > 1): ?>
<nav><ul class="pagination lp-pagination justify-content-center mb-0">
    <li class="page-item <?php echo !$pager['has_prev'] ? 'disabled' : ''; ?>">
        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] - 1])); ?>"><i class="bi bi-chevron-left"></i></a></li>
    <?php
    $rng = 2; $s = max(1, $pager['current']-$rng); $e = min($pager['total_pages'], $pager['current']+$rng);
    if ($s > 1): ?><li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>1])); ?>">1</a></li><?php if ($s > 2): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; endif;
    for ($p=$s; $p<=$e; $p++): ?><li class="page-item <?php echo $p===$pager['current']?'active':''; ?>"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>$p])); ?>"><?php echo $p; ?></a></li><?php endfor;
    if ($e < $pager['total_pages']): if ($e < $pager['total_pages']-1): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; ?><li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>$pager['total_pages']])); ?>"><?php echo $pager['total_pages']; ?></a></li><?php endif; ?>
    <li class="page-item <?php echo !$pager['has_next'] ? 'disabled' : ''; ?>">
        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] + 1])); ?>"><i class="bi bi-chevron-right"></i></a></li>
</ul></nav>
<?php endif; ?>

<?php endif; ?>

<!-- ═══ Create / Edit Modal ═════════════════════════════════════ -->
<div class="modal fade" id="incomeModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content" style="border:0; border-radius:var(--lp-radius-lg); overflow:hidden">
            <form method="POST" action="income.php" class="lp-form" novalidate>
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="save">
                <input type="hidden" name="edit_id" id="inc_edit_id" value="0">

                <div class="modal-header" style="background:var(--lp-snow); border-bottom:1px solid var(--lp-silver)">
                    <h5 class="modal-title" id="incomeModalLabel">
                        <i class="bi bi-graph-up-arrow me-1"></i> Add Income
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Category <span class="text-danger">*</span></label>
                            <select class="form-select" name="category" id="inc_category" required>
                                <option value="">Select…</option>
                                <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo h($cat); ?>"><?php echo h($cat); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Client <small class="text-muted">(optional)</small></label>
                            <select class="form-select" name="client_id" id="inc_client_id">
                                <option value="0">— None —</option>
                                <?php foreach ($client_list as $cl): ?>
                                <option value="<?php echo (int)$cl['id']; ?>"><?php echo h($cl['company_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Description <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="description" id="inc_description"
                                   required maxlength="255" placeholder="What is this income from?">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Amount <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><?php echo CURRENCY_SYMBOL; ?></span>
                                <input type="number" class="form-control" name="amount" id="inc_amount"
                                       step="0.01" min="0.01" required placeholder="0.00">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Date <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" name="income_date" id="inc_date"
                                   value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Reference</label>
                            <input type="text" class="form-control" name="reference" id="inc_reference"
                                   maxlength="100" placeholder="e.g. TRX-1234">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Notes</label>
                            <textarea class="form-control" name="notes" id="inc_notes" rows="2"
                                      style="resize:vertical" placeholder="Optional notes…"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer" style="background:var(--lp-snow); border-top:1px solid var(--lp-silver)">
                    <button type="button" class="btn btn-lp-outline" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-lp-primary">
                        <i class="bi bi-check-lg me-1"></i> <span id="incSaveLabel">Save</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function resetIncomeForm() {
    document.getElementById('inc_edit_id').value    = '0';
    document.getElementById('inc_category').value   = '';
    document.getElementById('inc_client_id').value  = '0';
    document.getElementById('inc_description').value= '';
    document.getElementById('inc_amount').value     = '';
    document.getElementById('inc_date').value       = '<?php echo date('Y-m-d'); ?>';
    document.getElementById('inc_reference').value  = '';
    document.getElementById('inc_notes').value      = '';
    document.getElementById('incomeModalLabel').innerHTML = '<i class="bi bi-graph-up-arrow me-1"></i> Add Income';
    document.getElementById('incSaveLabel').textContent   = 'Save';
}

function editIncome(data) {
    document.getElementById('inc_edit_id').value     = data.id;
    document.getElementById('inc_category').value    = data.category;
    document.getElementById('inc_client_id').value   = data.client_id || '0';
    document.getElementById('inc_description').value = data.description;
    document.getElementById('inc_amount').value      = parseFloat(data.amount).toFixed(2);
    document.getElementById('inc_date').value        = data.income_date;
    document.getElementById('inc_reference').value   = data.reference || '';
    document.getElementById('inc_notes').value       = data.notes || '';
    document.getElementById('incomeModalLabel').innerHTML = '<i class="bi bi-pencil me-1"></i> Edit Income';
    document.getElementById('incSaveLabel').textContent   = 'Update';
    new bootstrap.Modal(document.getElementById('incomeModal')).show();
}
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
