<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  footer.php — Shared Page Footer & Scripts
 * ═══════════════════════════════════════════════════════════════
 *
 *  Include at the bottom of every page:
 *    require_once 'footer.php';
 *
 *  This file:
 *    1. Closes the .lp-page container opened in header.php
 *    2. Renders the site footer with copyright
 *    3. Loads Bootstrap 5 JS bundle
 *    4. Provides utility scripts used across all pages:
 *       - Confirm-before-delete on any [data-confirm] element
 *       - Auto-dismiss flash messages after 6 seconds
 *       - Tooltip initialization
 *       - Print button handler
 *       - CSRF-aware fetch helper for future AJAX calls
 */
?>

</div><!-- /.container.lp-page  (opened in header.php) -->

<!-- ═══ Site Footer ═════════════════════════════════════════════ -->
<footer class="lp-footer">
    <div class="container">
        <div class="lp-footer-inner">
            <!-- Left: Brand + copyright -->
            <div class="lp-footer-left">
                <span class="lp-footer-brand">
                    <i class="bi bi-journal-bookmark-fill"></i>
                    <?php echo APP_NAME; ?>
                </span>
                <span class="lp-footer-copy">
                    &copy; <?php echo date('Y'); ?> <?php echo APP_NAME; ?>. All rights reserved.
                </span>
            </div>

            <!-- Right: Version + links -->
            <div class="lp-footer-right">
                <span class="lp-footer-version">v<?php echo APP_VERSION; ?></span>
                <?php if (is_logged_in()): ?>
                <span class="lp-footer-sep">&middot;</span>
                <a href="<?php echo APP_URL; ?>/profile.php" class="lp-footer-link">Profile</a>
                <span class="lp-footer-sep">&middot;</span>
                <a href="<?php echo APP_URL; ?>/dashboard.php" class="lp-footer-link">Dashboard</a>
                <?php endif; ?>
            </div>
        </div>
    </div>
</footer>

<!-- ── Footer Styles ──────────────────────────────────────────── -->
<style>
    .lp-footer {
        background: var(--lp-white);
        border-top: 1px solid var(--lp-silver);
        padding: 1.25rem 0;
        margin-top: auto;
        font-size: .8rem;
    }
    .lp-footer-inner {
        display: flex;
        flex-wrap: wrap;
        align-items: center;
        justify-content: space-between;
        gap: .75rem;
    }
    .lp-footer-left {
        display: flex;
        align-items: center;
        gap: .75rem;
        flex-wrap: wrap;
    }
    .lp-footer-brand {
        font-family: var(--lp-font-display);
        font-size: .95rem;
        color: var(--lp-navy);
        display: inline-flex;
        align-items: center;
        gap: .35rem;
    }
    .lp-footer-copy {
        color: var(--lp-slate-mid);
    }
    .lp-footer-right {
        display: flex;
        align-items: center;
        gap: .5rem;
        color: var(--lp-slate-light);
    }
    .lp-footer-version {
        font-family: 'DM Sans', monospace;
        font-size: .72rem;
        background: var(--lp-snow);
        padding: .15rem .5rem;
        border-radius: 50rem;
        color: var(--lp-slate-mid);
        border: 1px solid var(--lp-silver);
    }
    .lp-footer-sep {
        color: var(--lp-silver);
    }
    .lp-footer-link {
        color: var(--lp-slate-mid);
        text-decoration: none;
        transition: var(--lp-transition);
    }
    .lp-footer-link:hover {
        color: var(--lp-navy);
    }

    /* Ensure footer stays at bottom on short pages */
    body {
        display: flex;
        flex-direction: column;
        min-height: 100vh;
    }
    .lp-page {
        flex: 1;
    }

    @media (max-width: 576px) {
        .lp-footer-inner {
            flex-direction: column;
            text-align: center;
        }
        .lp-footer-left,
        .lp-footer-right {
            justify-content: center;
        }
    }
</style>

<!-- ── Bootstrap 5.3 JS Bundle (includes Popper) ──────────────── -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>

<!-- ── Application Utility Scripts ─────────────────────────────── -->
<script>
document.addEventListener('DOMContentLoaded', function() {

    /* ── 1. Confirm-before-action ──────────────────────────────
     *  Add data-confirm="Are you sure?" to any link, button, or
     *  form to show a native confirm dialog before proceeding.
     *
     *  <a href="delete.php?id=5" data-confirm="Delete this invoice?">Delete</a>
     *  <form method="post" data-confirm="Submit payment?">
     */
    document.querySelectorAll('[data-confirm]').forEach(function(el) {
        el.addEventListener('click', function(e) {
            if (!confirm(this.getAttribute('data-confirm'))) {
                e.preventDefault();
                e.stopImmediatePropagation();
            }
        });

        /* If it's a form, also intercept submit */
        if (el.tagName === 'FORM') {
            el.addEventListener('submit', function(e) {
                if (!confirm(this.getAttribute('data-confirm'))) {
                    e.preventDefault();
                }
            });
        }
    });


    /* ── 2. Auto-dismiss flash messages after 6s ───────────────
     *  Fades out and removes the element from the DOM.
     */
    document.querySelectorAll('.lp-flash').forEach(function(flash) {
        setTimeout(function() {
            flash.style.transition = 'opacity .4s ease, transform .4s ease';
            flash.style.opacity = '0';
            flash.style.transform = 'translateY(-.5rem)';
            setTimeout(function() { flash.remove(); }, 400);
        }, 6000);
    });


    /* ── 3. Bootstrap Tooltip Initialization ───────────────────
     *  Add data-bs-toggle="tooltip" title="Helpful tip" to any
     *  element to activate Bootstrap tooltips.
     */
    var tooltipEls = document.querySelectorAll('[data-bs-toggle="tooltip"]');
    tooltipEls.forEach(function(el) {
        new bootstrap.Tooltip(el);
    });


    /* ── 4. Print Button Handler ───────────────────────────────
     *  Add class="lp-print-btn" to any button to trigger print.
     */
    document.querySelectorAll('.lp-print-btn').forEach(function(btn) {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            window.print();
        });
    });


    /* ── 5. Number Input Formatting ────────────────────────────
     *  Inputs with data-currency="true" will format to 2 decimal
     *  places on blur and prevent non-numeric input.
     */
    document.querySelectorAll('[data-currency]').forEach(function(input) {
        input.addEventListener('blur', function() {
            var val = parseFloat(this.value);
            if (!isNaN(val)) {
                this.value = val.toFixed(2);
            }
        });
        input.addEventListener('keypress', function(e) {
            var char = String.fromCharCode(e.which);
            if (!/[\d.]/.test(char) && e.which !== 8) {
                e.preventDefault();
            }
        });
    });


    /* ── 6. Dynamic Line-Item Total Calculator ─────────────────
     *  For invoice forms: recalculates line_total and invoice
     *  totals when quantity or unit_price changes.
     *
     *  Inputs need: data-calc="qty" / data-calc="price"
     *  Output cell: data-calc="line-total"
     *  Subtotal:    #calc-subtotal
     *  Tax amount:  #calc-tax
     *  Total:       #calc-total
     */
    function recalcLineItems() {
        var subtotal = 0;
        document.querySelectorAll('.lp-line-item-row').forEach(function(row) {
            var qty   = parseFloat(row.querySelector('[data-calc="qty"]')?.value)   || 0;
            var price = parseFloat(row.querySelector('[data-calc="price"]')?.value) || 0;
            var total = qty * price;
            subtotal += total;

            var totalCell = row.querySelector('[data-calc="line-total"]');
            if (totalCell) {
                totalCell.textContent = total.toFixed(2);
            }
        });

        var taxRateEl = document.getElementById('tax_rate');
        var taxRate   = taxRateEl ? (parseFloat(taxRateEl.value) || 0) : 0;
        var discEl    = document.getElementById('discount_amount');
        var discount  = discEl ? (parseFloat(discEl.value) || 0) : 0;
        var taxAmount = subtotal * (taxRate / 100);
        var grandTotal = subtotal + taxAmount - discount;

        var subEl   = document.getElementById('calc-subtotal');
        var taxEl   = document.getElementById('calc-tax');
        var totalEl = document.getElementById('calc-total');

        if (subEl)   subEl.textContent   = subtotal.toFixed(2);
        if (taxEl)   taxEl.textContent   = taxAmount.toFixed(2);
        if (totalEl) totalEl.textContent = grandTotal.toFixed(2);
    }

    /* Attach listeners if invoice line-item rows exist */
    document.querySelectorAll('.lp-line-item-row [data-calc="qty"], .lp-line-item-row [data-calc="price"]').forEach(function(input) {
        input.addEventListener('input', recalcLineItems);
    });

    var taxRateEl = document.getElementById('tax_rate');
    if (taxRateEl) taxRateEl.addEventListener('input', recalcLineItems);

    var discEl = document.getElementById('discount_amount');
    if (discEl) discEl.addEventListener('input', recalcLineItems);

    /* Initial calculation on page load */
    if (document.querySelector('.lp-line-item-row')) {
        recalcLineItems();
    }


    /* ── 7. CSRF-aware Fetch Helper ────────────────────────────
     *  For future AJAX calls. Automatically includes the CSRF
     *  token from the meta tag or hidden input.
     *
     *  Usage:
     *    lpFetch('/api/endpoint', { method: 'POST', body: data })
     *      .then(res => res.json())
     *      .then(json => console.log(json));
     */
    window.lpFetch = function(url, options) {
        options = options || {};
        options.headers = options.headers || {};

        var tokenMeta  = document.querySelector('meta[name="csrf-token"]');
        var tokenInput = document.querySelector('input[name="csrf_token"]');
        var token      = tokenMeta ? tokenMeta.content : (tokenInput ? tokenInput.value : '');

        if (token) {
            options.headers['X-CSRF-Token'] = token;
        }

        return fetch(url, options);
    };

});
</script>

</body>
</html>
