<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  dashboard.php — Role-Aware Financial Dashboard
 * ═══════════════════════════════════════════════════════════════
 *
 *  Accountant sees:
 *    - 6 KPI stat cards (revenue, expenses, profit, receivables,
 *      overdue, active clients)
 *    - Recent invoices table
 *    - Recent payments table
 *    - Overdue invoice alerts
 *    - Monthly income vs expenses bar chart
 *
 *  Client sees:
 *    - 4 stat cards (total invoiced, total paid, balance due,
 *      invoice count)
 *    - Their own recent invoices
 *    - Their own recent payments
 *
 *  Every query is scoped by role. Clients can never see data
 *  belonging to other clients.
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

/* ── Auth guard ───────────────────────────────────────────────── */
if (session_status() === PHP_SESSION_NONE) session_start();
require_login();

$page_title = 'Dashboard';

/* ══════════════════════════════════════════════════════════════
 *  LOAD DATA — Accountant View
 * ══════════════════════════════════════════════════════════════ */
if (is_accountant()) {

    /* ── KPI Stats ────────────────────────────────────────── */

    // Total revenue: sum of all paid invoices
    $total_revenue = (float) db_scalar(
        "SELECT COALESCE(SUM(amount_paid), 0) FROM invoices WHERE status = 'paid'"
    );

    // Total expenses
    $total_expenses = (float) db_scalar(
        "SELECT COALESCE(SUM(amount), 0) FROM expenses"
    );

    // Net profit
    $net_profit = $total_revenue - $total_expenses;

    // Outstanding receivables (balance_due on non-cancelled invoices)
    $total_receivable = (float) db_scalar(
        "SELECT COALESCE(SUM(balance_due), 0) FROM invoices
         WHERE status NOT IN ('paid','cancelled')"
    );

    // Overdue count
    $overdue_count = (int) db_scalar(
        "SELECT COUNT(*) FROM invoices
         WHERE status = 'overdue'
            OR (status IN ('sent','draft') AND due_date < CURDATE())"
    );

    // Active clients
    $active_clients = (int) db_scalar(
        "SELECT COUNT(*) FROM clients WHERE is_active = 1"
    );

    /* ── Recent Invoices (last 7) ─────────────────────────── */
    $recent_invoices = db_select(
        "SELECT i.id, i.invoice_number, i.status, i.total, i.balance_due,
                i.due_date, c.company_name
         FROM invoices i
         JOIN clients c ON c.id = i.client_id
         ORDER BY i.created_at DESC
         LIMIT 7"
    );

    /* ── Recent Payments (last 7) ─────────────────────────── */
    $recent_payments = db_select(
        "SELECT p.id, p.amount, p.payment_date, p.method,
                i.invoice_number, c.company_name
         FROM payments p
         JOIN invoices i ON i.id = p.invoice_id
         JOIN clients c  ON c.id = p.client_id
         ORDER BY p.created_at DESC
         LIMIT 7"
    );

    /* ── Overdue Invoices ─────────────────────────────────── */
    $overdue_invoices = db_select(
        "SELECT i.id, i.invoice_number, i.total, i.balance_due,
                i.due_date, c.company_name
         FROM invoices i
         JOIN clients c ON c.id = i.client_id
         WHERE i.status = 'overdue'
            OR (i.status IN ('sent','draft') AND i.due_date < CURDATE())
         ORDER BY i.due_date ASC
         LIMIT 5"
    );

    /* ── Monthly Chart Data (last 6 months) ───────────────── */
    $chart_months = [];
    $chart_income = [];
    $chart_expense = [];

    for ($m = 5; $m >= 0; $m--) {
        $month_start = date('Y-m-01', strtotime("-{$m} months"));
        $month_end   = date('Y-m-t',  strtotime("-{$m} months"));
        $label       = date('M Y',    strtotime("-{$m} months"));

        $chart_months[]  = $label;

        $chart_income[]  = (float) db_scalar(
            "SELECT COALESCE(SUM(amount_paid), 0) FROM invoices
             WHERE status = 'paid'
               AND issue_date BETWEEN ? AND ?",
            ['ss', $month_start, $month_end]
        ) + (float) db_scalar(
            "SELECT COALESCE(SUM(amount), 0) FROM income
             WHERE income_date BETWEEN ? AND ?",
            ['ss', $month_start, $month_end]
        );

        $chart_expense[] = (float) db_scalar(
            "SELECT COALESCE(SUM(amount), 0) FROM expenses
             WHERE expense_date BETWEEN ? AND ?",
            ['ss', $month_start, $month_end]
        );
    }

/* ══════════════════════════════════════════════════════════════
 *  LOAD DATA — Client View
 * ══════════════════════════════════════════════════════════════ */
} else {

    $cid = (int) ($_SESSION['client_id'] ?? 0);

    // Total invoiced to this client
    $client_total_invoiced = (float) db_scalar(
        "SELECT COALESCE(SUM(total), 0) FROM invoices
         WHERE client_id = ? AND status != 'cancelled'",
        ['i', $cid]
    );

    // Total paid by this client
    $client_total_paid = (float) db_scalar(
        "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE client_id = ?",
        ['i', $cid]
    );

    // Balance due
    $client_balance_due = (float) db_scalar(
        "SELECT COALESCE(SUM(balance_due), 0) FROM invoices
         WHERE client_id = ? AND status NOT IN ('paid','cancelled')",
        ['i', $cid]
    );

    // Invoice count
    $client_invoice_count = (int) db_scalar(
        "SELECT COUNT(*) FROM invoices WHERE client_id = ?",
        ['i', $cid]
    );

    // Recent invoices
    $recent_invoices = db_select(
        "SELECT id, invoice_number, status, total, balance_due, due_date
         FROM invoices
         WHERE client_id = ?
         ORDER BY created_at DESC
         LIMIT 7",
        ['i', $cid]
    );

    // Recent payments
    $recent_payments = db_select(
        "SELECT p.id, p.amount, p.payment_date, p.method, i.invoice_number
         FROM payments p
         JOIN invoices i ON i.id = p.invoice_id
         WHERE p.client_id = ?
         ORDER BY p.created_at DESC
         LIMIT 7",
        ['i', $cid]
    );
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1>
            <?php if (is_accountant()): ?>
                Dashboard
            <?php else: ?>
                My Dashboard
            <?php endif; ?>
        </h1>
        <div class="lp-subtitle">
            <?php echo date('l, F j, Y'); ?> &mdash;
            Welcome back, <?php echo h($_SESSION['user_name']); ?>
        </div>
    </div>
    <?php if (is_accountant()): ?>
    <div class="d-flex gap-2 flex-wrap">
        <a href="<?php echo APP_URL; ?>/invoice_form.php" class="btn btn-lp-primary">
            <i class="bi bi-plus-lg me-1"></i> New Invoice
        </a>
        <a href="<?php echo APP_URL; ?>/client_form.php" class="btn btn-lp-outline">
            <i class="bi bi-person-plus me-1"></i> Add Client
        </a>
    </div>
    <?php endif; ?>
</div>


<?php if (is_accountant()): ?>
<!-- ═══════════════════════════════════════════════════════════════
     ACCOUNTANT DASHBOARD
     ═══════════════════════════════════════════════════════════════ -->

<!-- ── KPI Stat Cards ──────────────────────────────────────────── -->
<div class="row g-3 mb-4">
    <!-- Revenue -->
    <div class="col-sm-6 col-lg-4 col-xl-2">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon emerald"><i class="bi bi-graph-up-arrow"></i></div>
                <div>
                    <div class="lp-stat-label">Revenue</div>
                    <div class="lp-stat-value"><?php echo format_currency($total_revenue); ?></div>
                </div>
            </div>
        </div>
    </div>
    <!-- Expenses -->
    <div class="col-sm-6 col-lg-4 col-xl-2">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon red"><i class="bi bi-wallet2"></i></div>
                <div>
                    <div class="lp-stat-label">Expenses</div>
                    <div class="lp-stat-value"><?php echo format_currency($total_expenses); ?></div>
                </div>
            </div>
        </div>
    </div>
    <!-- Net Profit -->
    <div class="col-sm-6 col-lg-4 col-xl-2">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon <?php echo $net_profit >= 0 ? 'emerald' : 'red'; ?>">
                    <i class="bi bi-<?php echo $net_profit >= 0 ? 'trophy' : 'arrow-down-circle'; ?>"></i>
                </div>
                <div>
                    <div class="lp-stat-label">Net Profit</div>
                    <div class="lp-stat-value"><?php echo format_currency($net_profit); ?></div>
                </div>
            </div>
        </div>
    </div>
    <!-- Receivable -->
    <div class="col-sm-6 col-lg-4 col-xl-2">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon amber"><i class="bi bi-hourglass-split"></i></div>
                <div>
                    <div class="lp-stat-label">Receivable</div>
                    <div class="lp-stat-value"><?php echo format_currency($total_receivable); ?></div>
                </div>
            </div>
        </div>
    </div>
    <!-- Overdue -->
    <div class="col-sm-6 col-lg-4 col-xl-2">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon red"><i class="bi bi-exclamation-triangle"></i></div>
                <div>
                    <div class="lp-stat-label">Overdue</div>
                    <div class="lp-stat-value"><?php echo $overdue_count; ?></div>
                </div>
            </div>
        </div>
    </div>
    <!-- Clients -->
    <div class="col-sm-6 col-lg-4 col-xl-2">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon purple"><i class="bi bi-people"></i></div>
                <div>
                    <div class="lp-stat-label">Clients</div>
                    <div class="lp-stat-value"><?php echo $active_clients; ?></div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- ── Overdue Alerts ──────────────────────────────────────────── -->
<?php if (!empty($overdue_invoices)): ?>
<div class="lp-card mb-4" style="border-left: 4px solid var(--lp-red);">
    <div class="card-header d-flex align-items-center gap-2">
        <i class="bi bi-exclamation-triangle-fill text-danger"></i>
        Overdue Invoices
    </div>
    <div class="card-body p-0">
        <div class="lp-table-responsive">
            <table class="lp-table">
                <thead>
                    <tr>
                        <th>Invoice</th>
                        <th>Client</th>
                        <th>Due Date</th>
                        <th class="text-end">Balance</th>
                        <th class="text-end">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($overdue_invoices as $oi): ?>
                    <tr>
                        <td>
                            <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$oi['id']; ?>"
                               class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                                <?php echo h($oi['invoice_number']); ?>
                            </a>
                        </td>
                        <td><?php echo h($oi['company_name']); ?></td>
                        <td>
                            <span class="text-danger fw-semibold">
                                <?php echo date(DATE_FORMAT, strtotime($oi['due_date'])); ?>
                            </span>
                            <small class="text-muted d-block" style="font-size:.72rem">
                                <?php
                                    $days = (int) round((time() - strtotime($oi['due_date'])) / 86400);
                                    echo $days . ' day' . ($days !== 1 ? 's' : '') . ' overdue';
                                ?>
                            </small>
                        </td>
                        <td class="text-end fw-semibold"><?php echo format_currency((float)$oi['balance_due']); ?></td>
                        <td class="text-end">
                            <a href="<?php echo APP_URL; ?>/payment_form.php?invoice_id=<?php echo (int)$oi['id']; ?>"
                               class="btn btn-sm btn-lp-primary">
                                Record Payment
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- ── Income vs Expenses Chart + Recent Activity ──────────────── -->
<div class="row g-3 mb-4">
    <!-- Chart -->
    <div class="col-lg-7">
        <div class="lp-card h-100">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-bar-chart-line me-1"></i> Income vs Expenses</span>
                <small class="text-muted">Last 6 Months</small>
            </div>
            <div class="card-body">
                <canvas id="incomeExpenseChart" height="260"></canvas>
            </div>
        </div>
    </div>

    <!-- Recent Payments -->
    <div class="col-lg-5">
        <div class="lp-card h-100">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-credit-card me-1"></i> Recent Payments</span>
                <a href="<?php echo APP_URL; ?>/payments.php" class="text-decoration-none"
                   style="font-size:.8rem; font-weight:500;">View All</a>
            </div>
            <div class="card-body p-0">
                <?php if (empty($recent_payments)): ?>
                    <div class="lp-empty">
                        <i class="bi bi-credit-card d-block"></i>
                        <p>No payments recorded yet.</p>
                    </div>
                <?php else: ?>
                <div class="list-group list-group-flush">
                    <?php foreach ($recent_payments as $rp): ?>
                    <div class="list-group-item border-0 px-3 py-2" style="border-bottom:1px solid #f1f5f9 !important">
                        <div class="d-flex align-items-center justify-content-between">
                            <div>
                                <div class="fw-semibold" style="font-size:.84rem; color:var(--lp-navy)">
                                    <?php echo h($rp['company_name']); ?>
                                </div>
                                <small class="text-muted">
                                    <?php echo h($rp['invoice_number']); ?> &middot;
                                    <?php echo date(DATE_FORMAT, strtotime($rp['payment_date'])); ?>
                                </small>
                            </div>
                            <div class="text-end">
                                <div class="fw-semibold" style="font-size:.88rem; color:var(--lp-emerald)">
                                    +<?php echo format_currency((float)$rp['amount']); ?>
                                </div>
                                <small class="text-muted text-capitalize" style="font-size:.72rem">
                                    <?php echo str_replace('_', ' ', h($rp['method'])); ?>
                                </small>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- ── Recent Invoices ─────────────────────────────────────────── -->
<div class="lp-card mb-4">
    <div class="card-header d-flex align-items-center justify-content-between">
        <span><i class="bi bi-receipt me-1"></i> Recent Invoices</span>
        <a href="<?php echo APP_URL; ?>/invoices.php" class="text-decoration-none"
           style="font-size:.8rem; font-weight:500;">View All</a>
    </div>
    <div class="card-body p-0">
        <?php if (empty($recent_invoices)): ?>
            <div class="lp-empty">
                <i class="bi bi-receipt d-block"></i>
                <p>No invoices created yet.</p>
            </div>
        <?php else: ?>
        <div class="lp-table-responsive">
            <table class="lp-table">
                <thead>
                    <tr>
                        <th>Invoice #</th>
                        <th>Client</th>
                        <th>Status</th>
                        <th>Due Date</th>
                        <th class="text-end">Total</th>
                        <th class="text-end">Balance</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recent_invoices as $ri): ?>
                    <tr>
                        <td>
                            <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$ri['id']; ?>"
                               class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                                <?php echo h($ri['invoice_number']); ?>
                            </a>
                        </td>
                        <td><?php echo h($ri['company_name']); ?></td>
                        <td><span class="lp-badge <?php echo h($ri['status']); ?>"><?php echo h(ucfirst($ri['status'])); ?></span></td>
                        <td><?php echo date(DATE_FORMAT, strtotime($ri['due_date'])); ?></td>
                        <td class="text-end"><?php echo format_currency((float)$ri['total']); ?></td>
                        <td class="text-end fw-semibold">
                            <?php echo format_currency((float)$ri['balance_due']); ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- ── Chart.js via CDN ────────────────────────────────────────── -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.7/dist/chart.umd.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var ctx = document.getElementById('incomeExpenseChart');
    if (!ctx) return;

    var months   = <?php echo json_encode($chart_months); ?>;
    var income   = <?php echo json_encode($chart_income); ?>;
    var expenses = <?php echo json_encode($chart_expense); ?>;

    /* Compute the computed CSS variable values */
    var style    = getComputedStyle(document.documentElement);
    var emerald  = style.getPropertyValue('--lp-emerald').trim()  || '#059669';
    var red      = style.getPropertyValue('--lp-red').trim()      || '#dc2626';
    var silver   = style.getPropertyValue('--lp-silver').trim()   || '#e2e8f0';
    var slateMid = style.getPropertyValue('--lp-slate-mid').trim()|| '#64748b';

    new Chart(ctx, {
        type: 'bar',
        data: {
            labels: months,
            datasets: [
                {
                    label: 'Income',
                    data: income,
                    backgroundColor: emerald + '22',
                    borderColor: emerald,
                    borderWidth: 2,
                    borderRadius: 6,
                    borderSkipped: false,
                },
                {
                    label: 'Expenses',
                    data: expenses,
                    backgroundColor: red + '22',
                    borderColor: red,
                    borderWidth: 2,
                    borderRadius: 6,
                    borderSkipped: false,
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            interaction: { mode: 'index', intersect: false },
            plugins: {
                legend: {
                    position: 'top',
                    labels: {
                        usePointStyle: true,
                        pointStyle: 'circle',
                        boxWidth: 8,
                        padding: 18,
                        font: { family: "'DM Sans', sans-serif", size: 12 },
                        color: slateMid
                    }
                },
                tooltip: {
                    backgroundColor: '#0f172a',
                    titleFont: { family: "'DM Sans', sans-serif" },
                    bodyFont:  { family: "'DM Sans', sans-serif" },
                    padding: 12,
                    cornerRadius: 8,
                    callbacks: {
                        label: function(ctx) {
                            return ctx.dataset.label + ': $' + ctx.parsed.y.toLocaleString(undefined, {minimumFractionDigits:2});
                        }
                    }
                }
            },
            scales: {
                x: {
                    grid: { display: false },
                    ticks: {
                        font: { family: "'DM Sans', sans-serif", size: 11 },
                        color: slateMid
                    }
                },
                y: {
                    beginAtZero: true,
                    grid: { color: silver },
                    ticks: {
                        font: { family: "'DM Sans', sans-serif", size: 11 },
                        color: slateMid,
                        callback: function(val) { return '$' + val.toLocaleString(); }
                    }
                }
            }
        }
    });
});
</script>


<?php else: ?>
<!-- ═══════════════════════════════════════════════════════════════
     CLIENT DASHBOARD
     ═══════════════════════════════════════════════════════════════ -->

<!-- ── Client Stat Cards ───────────────────────────────────────── -->
<div class="row g-3 mb-4">
    <div class="col-sm-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon blue"><i class="bi bi-receipt"></i></div>
                <div>
                    <div class="lp-stat-label">Total Invoiced</div>
                    <div class="lp-stat-value"><?php echo format_currency($client_total_invoiced); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-sm-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon emerald"><i class="bi bi-check-circle"></i></div>
                <div>
                    <div class="lp-stat-label">Total Paid</div>
                    <div class="lp-stat-value"><?php echo format_currency($client_total_paid); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-sm-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon amber"><i class="bi bi-hourglass-split"></i></div>
                <div>
                    <div class="lp-stat-label">Balance Due</div>
                    <div class="lp-stat-value"><?php echo format_currency($client_balance_due); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-sm-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon purple"><i class="bi bi-file-earmark-text"></i></div>
                <div>
                    <div class="lp-stat-label">Invoices</div>
                    <div class="lp-stat-value"><?php echo $client_invoice_count; ?></div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- ── Client: Recent Invoices ─────────────────────────────────── -->
<div class="row g-3 mb-4">
    <div class="col-lg-7">
        <div class="lp-card h-100">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-receipt me-1"></i> My Invoices</span>
                <a href="<?php echo APP_URL; ?>/invoices.php" class="text-decoration-none"
                   style="font-size:.8rem; font-weight:500;">View All</a>
            </div>
            <div class="card-body p-0">
                <?php if (empty($recent_invoices)): ?>
                    <div class="lp-empty">
                        <i class="bi bi-receipt d-block"></i>
                        <p>No invoices yet.</p>
                    </div>
                <?php else: ?>
                <div class="lp-table-responsive">
                    <table class="lp-table">
                        <thead>
                            <tr>
                                <th>Invoice #</th>
                                <th>Status</th>
                                <th>Due Date</th>
                                <th class="text-end">Total</th>
                                <th class="text-end">Balance</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_invoices as $ri): ?>
                            <tr>
                                <td>
                                    <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$ri['id']; ?>"
                                       class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                                        <?php echo h($ri['invoice_number']); ?>
                                    </a>
                                </td>
                                <td><span class="lp-badge <?php echo h($ri['status']); ?>"><?php echo h(ucfirst($ri['status'])); ?></span></td>
                                <td><?php echo date(DATE_FORMAT, strtotime($ri['due_date'])); ?></td>
                                <td class="text-end"><?php echo format_currency((float)$ri['total']); ?></td>
                                <td class="text-end fw-semibold"><?php echo format_currency((float)$ri['balance_due']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Client: Recent Payments -->
    <div class="col-lg-5">
        <div class="lp-card h-100">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-credit-card me-1"></i> My Payments</span>
                <a href="<?php echo APP_URL; ?>/payments.php" class="text-decoration-none"
                   style="font-size:.8rem; font-weight:500;">View All</a>
            </div>
            <div class="card-body p-0">
                <?php if (empty($recent_payments)): ?>
                    <div class="lp-empty">
                        <i class="bi bi-credit-card d-block"></i>
                        <p>No payments recorded yet.</p>
                    </div>
                <?php else: ?>
                <div class="list-group list-group-flush">
                    <?php foreach ($recent_payments as $rp): ?>
                    <div class="list-group-item border-0 px-3 py-2" style="border-bottom:1px solid #f1f5f9 !important">
                        <div class="d-flex align-items-center justify-content-between">
                            <div>
                                <div class="fw-semibold" style="font-size:.84rem; color:var(--lp-navy)">
                                    <?php echo h($rp['invoice_number']); ?>
                                </div>
                                <small class="text-muted">
                                    <?php echo date(DATE_FORMAT, strtotime($rp['payment_date'])); ?> &middot;
                                    <?php echo str_replace('_', ' ', h(ucfirst($rp['method']))); ?>
                                </small>
                            </div>
                            <div class="fw-semibold" style="font-size:.88rem; color:var(--lp-emerald)">
                                +<?php echo format_currency((float)$rp['amount']); ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php endif; ?>

<?php require_once __DIR__ . '/footer.php'; ?>
