<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  clients.php — Client Management (Accountant Only)
 * ═══════════════════════════════════════════════════════════════
 *
 *  Features:
 *    - Paginated client list with search by name/email
 *    - Filter by active / inactive status
 *    - Quick-view stats per client (invoices, balance)
 *    - Delete with CSRF-protected POST confirmation
 *    - Fully responsive: table on desktop, cards on mobile
 *
 *  Authorization:
 *    - Accountant role required. Clients never access this page.
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

/* ══════════════════════════════════════════════════════════════
 *  HANDLE DELETE (POST)
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
        redirect('clients.php');
    }

    $delete_id = (int) ($_POST['client_id'] ?? 0);

    if ($delete_id > 0) {
        /*
         * Check if client has any invoices — RESTRICT deletion if so.
         * This mirrors the FK constraint and gives a friendly message.
         */
        $invoice_count = (int) db_scalar(
            "SELECT COUNT(*) FROM invoices WHERE client_id = ?",
            ['i', $delete_id]
        );

        if ($invoice_count > 0) {
            flash('error', "Cannot delete this client — they have {$invoice_count} invoice(s) on record. Deactivate instead.");
        } else {
            /* Also remove any linked user account */
            db_execute("UPDATE users SET client_id = NULL WHERE client_id = ?", ['i', $delete_id]);
            db_execute("DELETE FROM clients WHERE id = ?", ['i', $delete_id]);
            flash('success', 'Client deleted successfully.');
        }
    }

    csrf_regenerate();
    redirect('clients.php');
}

/* ══════════════════════════════════════════════════════════════
 *  FILTERS & SEARCH
 * ══════════════════════════════════════════════════════════════ */
$search  = trim($_GET['search'] ?? '');
$status  = $_GET['status'] ?? 'all';   // all | active | inactive

/* Build WHERE clause dynamically */
$where   = [];
$params  = [];
$types   = '';

if ($search !== '') {
    $where[]  = "(c.company_name LIKE CONCAT('%', ?, '%')
                  OR c.contact_name LIKE CONCAT('%', ?, '%')
                  OR c.email LIKE CONCAT('%', ?, '%'))";
    $safe     = db_escape_like($search);
    $params[] = $safe;
    $params[] = $safe;
    $params[] = $safe;
    $types   .= 'sss';
}

if ($status === 'active') {
    $where[] = "c.is_active = 1";
} elseif ($status === 'inactive') {
    $where[] = "c.is_active = 0";
}

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

/* ── Pagination ───────────────────────────────────────────────── */
$count_params = empty($types) ? [] : array_merge([$types], $params);
$pager = db_paginate(
    "SELECT COUNT(*) FROM clients c {$where_sql}",
    $count_params
);

/* ── Fetch Clients ────────────────────────────────────────────── */
$order_sql = "ORDER BY c.company_name ASC";
$limit_sql = "LIMIT ? OFFSET ?";

$select_params = $params;
$select_types  = $types . 'ii';
$select_params[] = $pager['limit'];
$select_params[] = $pager['offset'];

$clients = db_select(
    "SELECT c.*,
            (SELECT COUNT(*) FROM invoices WHERE client_id = c.id) AS invoice_count,
            (SELECT COALESCE(SUM(balance_due), 0)
             FROM invoices WHERE client_id = c.id AND status NOT IN ('paid','cancelled')) AS balance_due
     FROM clients c
     {$where_sql}
     {$order_sql}
     {$limit_sql}",
    array_merge([$select_types], $select_params)
);

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Clients';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1><i class="bi bi-people me-2" style="font-size:1.4rem"></i>Clients</h1>
        <div class="lp-subtitle">
            <?php echo $pager['total_rows']; ?> client<?php echo $pager['total_rows'] !== 1 ? 's' : ''; ?> on record
        </div>
    </div>
    <a href="<?php echo APP_URL; ?>/client_form.php" class="btn btn-lp-primary">
        <i class="bi bi-plus-lg me-1"></i> Add Client
    </a>
</div>

<!-- ═══ Search & Filter Bar ═════════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-body py-3">
        <form method="GET" action="clients.php" class="row g-2 align-items-end">
            <!-- Search -->
            <div class="col-lg-5 col-md-6">
                <div class="input-group">
                    <span class="input-group-text bg-white border-end-0" style="border-color:var(--lp-silver)">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input type="text" name="search" class="form-control border-start-0 ps-0"
                           style="border-color:var(--lp-silver); font-size:.875rem"
                           placeholder="Search by company, contact, or email…"
                           value="<?php echo h($search); ?>">
                </div>
            </div>

            <!-- Status filter -->
            <div class="col-lg-3 col-md-4">
                <select name="status" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all"      <?php echo $status === 'all'      ? 'selected' : ''; ?>>All Statuses</option>
                    <option value="active"   <?php echo $status === 'active'   ? 'selected' : ''; ?>>Active</option>
                    <option value="inactive" <?php echo $status === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                </select>
            </div>

            <!-- Buttons -->
            <div class="col-lg-4 col-md-12 d-flex gap-2">
                <button type="submit" class="btn btn-lp-primary">
                    <i class="bi bi-funnel me-1"></i> Filter
                </button>
                <?php if ($search !== '' || $status !== 'all'): ?>
                <a href="<?php echo APP_URL; ?>/clients.php" class="btn btn-lp-outline">
                    <i class="bi bi-x-lg me-1"></i> Clear
                </a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- ═══ Client List ═════════════════════════════════════════════ -->
<?php if (empty($clients)): ?>
    <div class="lp-card">
        <div class="lp-empty">
            <i class="bi bi-people d-block"></i>
            <p>
                <?php echo ($search !== '' || $status !== 'all')
                    ? 'No clients match your search.'
                    : 'No clients yet. Add your first client to get started.'; ?>
            </p>
            <?php if ($search === '' && $status === 'all'): ?>
            <a href="<?php echo APP_URL; ?>/client_form.php" class="btn btn-lp-primary mt-3">
                <i class="bi bi-plus-lg me-1"></i> Add Client
            </a>
            <?php endif; ?>
        </div>
    </div>
<?php else: ?>

<!-- Desktop table (hidden on small screens) -->
<div class="lp-card mb-4 d-none d-md-block">
    <div class="lp-table-responsive">
        <table class="lp-table">
            <thead>
                <tr>
                    <th>Company</th>
                    <th>Contact</th>
                    <th>Email</th>
                    <th>Phone</th>
                    <th class="text-center">Invoices</th>
                    <th class="text-end">Balance Due</th>
                    <th class="text-center">Status</th>
                    <th class="text-end">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($clients as $c): ?>
                <tr>
                    <!-- Company -->
                    <td>
                        <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$c['id']; ?>"
                           class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                            <?php echo h($c['company_name']); ?>
                        </a>
                        <?php if ($c['tax_id']): ?>
                        <small class="d-block text-muted" style="font-size:.72rem">
                            Tax ID: <?php echo h($c['tax_id']); ?>
                        </small>
                        <?php endif; ?>
                    </td>

                    <!-- Contact -->
                    <td><?php echo h($c['contact_name']); ?></td>

                    <!-- Email -->
                    <td>
                        <a href="mailto:<?php echo h($c['email']); ?>" class="text-decoration-none"
                           style="color:var(--lp-blue); font-size:.84rem">
                            <?php echo h($c['email']); ?>
                        </a>
                    </td>

                    <!-- Phone -->
                    <td style="font-size:.84rem"><?php echo h($c['phone'] ?? '—'); ?></td>

                    <!-- Invoice count -->
                    <td class="text-center">
                        <span class="badge rounded-pill"
                              style="background:var(--lp-blue-soft); color:var(--lp-blue); font-size:.75rem">
                            <?php echo (int)$c['invoice_count']; ?>
                        </span>
                    </td>

                    <!-- Balance due -->
                    <td class="text-end fw-semibold">
                        <?php if ((float)$c['balance_due'] > 0): ?>
                            <span style="color:var(--lp-amber)"><?php echo format_currency((float)$c['balance_due']); ?></span>
                        <?php else: ?>
                            <span class="text-muted"><?php echo format_currency(0); ?></span>
                        <?php endif; ?>
                    </td>

                    <!-- Status -->
                    <td class="text-center">
                        <?php if ((int)$c['is_active']): ?>
                            <span class="lp-badge paid">Active</span>
                        <?php else: ?>
                            <span class="lp-badge cancelled">Inactive</span>
                        <?php endif; ?>
                    </td>

                    <!-- Actions -->
                    <td class="text-end">
                        <div class="dropdown">
                            <button class="btn btn-sm btn-lp-outline dropdown-toggle" type="button"
                                    data-bs-toggle="dropdown" aria-expanded="false"
                                    style="padding:.3rem .65rem; font-size:.78rem">
                                Actions
                            </button>
                            <ul class="dropdown-menu dropdown-menu-end shadow-sm">
                                <li>
                                    <a class="dropdown-item" href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$c['id']; ?>">
                                        <i class="bi bi-eye me-2"></i> View
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="<?php echo APP_URL; ?>/client_form.php?id=<?php echo (int)$c['id']; ?>">
                                        <i class="bi bi-pencil me-2"></i> Edit
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="<?php echo APP_URL; ?>/invoice_form.php?client_id=<?php echo (int)$c['id']; ?>">
                                        <i class="bi bi-receipt me-2"></i> New Invoice
                                    </a>
                                </li>
                                <li><hr class="dropdown-divider"></li>
                                <li>
                                    <form method="POST" action="clients.php" class="px-3 py-1"
                                          data-confirm="Delete &quot;<?php echo h($c['company_name']); ?>&quot;? This cannot be undone.">
                                        <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                                               value="<?php echo csrf_token(); ?>">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="client_id" value="<?php echo (int)$c['id']; ?>">
                                        <button type="submit" class="dropdown-item text-danger p-0">
                                            <i class="bi bi-trash me-2"></i> Delete
                                        </button>
                                    </form>
                                </li>
                            </ul>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Mobile card list (visible only on small screens) -->
<div class="d-md-none mb-4">
    <?php foreach ($clients as $c): ?>
    <div class="lp-card mb-3">
        <div class="card-body">
            <!-- Top row: company + status -->
            <div class="d-flex align-items-start justify-content-between mb-2">
                <div>
                    <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$c['id']; ?>"
                       class="fw-semibold text-decoration-none d-block" style="color:var(--lp-navy); font-size:.95rem">
                        <?php echo h($c['company_name']); ?>
                    </a>
                    <small class="text-muted"><?php echo h($c['contact_name']); ?></small>
                </div>
                <?php if ((int)$c['is_active']): ?>
                    <span class="lp-badge paid">Active</span>
                <?php else: ?>
                    <span class="lp-badge cancelled">Inactive</span>
                <?php endif; ?>
            </div>

            <!-- Contact details -->
            <div class="mb-3" style="font-size:.84rem">
                <div class="mb-1">
                    <i class="bi bi-envelope me-1 text-muted"></i>
                    <a href="mailto:<?php echo h($c['email']); ?>" class="text-decoration-none" style="color:var(--lp-blue)">
                        <?php echo h($c['email']); ?>
                    </a>
                </div>
                <?php if ($c['phone']): ?>
                <div>
                    <i class="bi bi-telephone me-1 text-muted"></i>
                    <?php echo h($c['phone']); ?>
                </div>
                <?php endif; ?>
            </div>

            <!-- Stats row -->
            <div class="d-flex gap-3 mb-3 pb-3" style="border-bottom:1px solid var(--lp-silver)">
                <div>
                    <small class="text-muted d-block" style="font-size:.7rem; text-transform:uppercase; letter-spacing:.04em">Invoices</small>
                    <span class="fw-semibold" style="color:var(--lp-navy)"><?php echo (int)$c['invoice_count']; ?></span>
                </div>
                <div>
                    <small class="text-muted d-block" style="font-size:.7rem; text-transform:uppercase; letter-spacing:.04em">Balance Due</small>
                    <span class="fw-semibold" style="color: <?php echo (float)$c['balance_due'] > 0 ? 'var(--lp-amber)' : 'var(--lp-slate-mid)'; ?>">
                        <?php echo format_currency((float)$c['balance_due']); ?>
                    </span>
                </div>
            </div>

            <!-- Action buttons -->
            <div class="d-flex gap-2 flex-wrap">
                <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$c['id']; ?>"
                   class="btn btn-sm btn-lp-outline">
                    <i class="bi bi-eye me-1"></i> View
                </a>
                <a href="<?php echo APP_URL; ?>/client_form.php?id=<?php echo (int)$c['id']; ?>"
                   class="btn btn-sm btn-lp-outline">
                    <i class="bi bi-pencil me-1"></i> Edit
                </a>
                <a href="<?php echo APP_URL; ?>/invoice_form.php?client_id=<?php echo (int)$c['id']; ?>"
                   class="btn btn-sm btn-lp-primary">
                    <i class="bi bi-receipt me-1"></i> Invoice
                </a>
            </div>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- ═══ Pagination ══════════════════════════════════════════════ -->
<?php if ($pager['total_pages'] > 1): ?>
<nav aria-label="Client pagination">
    <ul class="pagination lp-pagination justify-content-center mb-0">
        <!-- Previous -->
        <li class="page-item <?php echo !$pager['has_prev'] ? 'disabled' : ''; ?>">
            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] - 1])); ?>">
                <i class="bi bi-chevron-left"></i>
            </a>
        </li>

        <?php
        /*
         * Smart page range: show first, last, current ± 2, and ellipses.
         * This avoids rendering 50+ page links on large datasets.
         */
        $range = 2;
        $start = max(1, $pager['current'] - $range);
        $end   = min($pager['total_pages'], $pager['current'] + $range);

        if ($start > 1): ?>
            <li class="page-item">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => 1])); ?>">1</a>
            </li>
            <?php if ($start > 2): ?>
            <li class="page-item disabled"><span class="page-link">…</span></li>
            <?php endif; ?>
        <?php endif;

        for ($p = $start; $p <= $end; $p++): ?>
            <li class="page-item <?php echo $p === $pager['current'] ? 'active' : ''; ?>">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $p])); ?>">
                    <?php echo $p; ?>
                </a>
            </li>
        <?php endfor;

        if ($end < $pager['total_pages']): ?>
            <?php if ($end < $pager['total_pages'] - 1): ?>
            <li class="page-item disabled"><span class="page-link">…</span></li>
            <?php endif; ?>
            <li class="page-item">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['total_pages']])); ?>">
                    <?php echo $pager['total_pages']; ?>
                </a>
            </li>
        <?php endif; ?>

        <!-- Next -->
        <li class="page-item <?php echo !$pager['has_next'] ? 'disabled' : ''; ?>">
            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] + 1])); ?>">
                <i class="bi bi-chevron-right"></i>
            </a>
        </li>
    </ul>
</nav>
<?php endif; ?>

<?php endif; /* end if !empty($clients) */ ?>

<?php require_once __DIR__ . '/footer.php'; ?>
