<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  client_view.php — Client Detail / Profile View
 * ═══════════════════════════════════════════════════════════════
 *
 *  Displays full client profile:
 *    - Contact details + address card
 *    - 4 financial stat cards
 *    - Invoice history table with status badges
 *    - Payment history list
 *    - Quick actions: edit, new invoice, record payment
 *
 *  Authorization:
 *    - Accountant: can view any client
 *    - Client: can only view their own profile (via session client_id)
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_login();

/* ══════════════════════════════════════════════════════════════
 *  LOAD CLIENT
 * ══════════════════════════════════════════════════════════════ */
$client_id = isset($_GET['id']) ? (int) $_GET['id'] : 0;

/* Client role: force to their own profile */
if (is_client()) {
    $client_id = (int) ($_SESSION['client_id'] ?? 0);
}

if ($client_id <= 0) {
    flash('error', 'Invalid client.');
    redirect(is_accountant() ? 'clients.php' : 'dashboard.php');
}

$client = db_select_one("SELECT * FROM clients WHERE id = ?", ['i', $client_id]);

if (!$client) {
    flash('error', 'Client not found.');
    redirect(is_accountant() ? 'clients.php' : 'dashboard.php');
}

/* ══════════════════════════════════════════════════════════════
 *  LOAD STATS & RELATED DATA
 * ══════════════════════════════════════════════════════════════ */

/* ── Financial Summary ────────────────────────────────────────── */
$stat_total_invoiced = (float) db_scalar(
    "SELECT COALESCE(SUM(total), 0) FROM invoices
     WHERE client_id = ? AND status != 'cancelled'",
    ['i', $client_id]
);

$stat_total_paid = (float) db_scalar(
    "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE client_id = ?",
    ['i', $client_id]
);

$stat_balance_due = (float) db_scalar(
    "SELECT COALESCE(SUM(balance_due), 0) FROM invoices
     WHERE client_id = ? AND status NOT IN ('paid','cancelled')",
    ['i', $client_id]
);

$stat_invoice_count = (int) db_scalar(
    "SELECT COUNT(*) FROM invoices WHERE client_id = ?",
    ['i', $client_id]
);

$stat_overdue_count = (int) db_scalar(
    "SELECT COUNT(*) FROM invoices
     WHERE client_id = ?
       AND (status = 'overdue' OR (status IN ('sent','draft') AND due_date < CURDATE()))",
    ['i', $client_id]
);

/* ── Invoices (most recent 10) ────────────────────────────────── */
$invoices = db_select(
    "SELECT id, invoice_number, status, issue_date, due_date,
            total, amount_paid, balance_due
     FROM invoices
     WHERE client_id = ?
     ORDER BY issue_date DESC
     LIMIT 10",
    ['i', $client_id]
);

/* ── Payments (most recent 10) ────────────────────────────────── */
$payments = db_select(
    "SELECT p.id, p.amount, p.payment_date, p.method, p.reference,
            i.invoice_number
     FROM payments p
     JOIN invoices i ON i.id = p.invoice_id
     WHERE p.client_id = ?
     ORDER BY p.payment_date DESC
     LIMIT 10",
    ['i', $client_id]
);

/* ── Linked user account ──────────────────────────────────────── */
$linked_user = db_select_one(
    "SELECT id, full_name, email, is_active, last_login_at
     FROM users WHERE client_id = ? AND role = 'client' LIMIT 1",
    ['i', $client_id]
);

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = $client['company_name'];
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div class="d-flex align-items-center gap-3">
        <!-- Avatar circle -->
        <div style="width:52px; height:52px; border-radius:var(--lp-radius-lg);
                    background:var(--lp-navy); color:var(--lp-white);
                    display:flex; align-items:center; justify-content:center;
                    font-family:var(--lp-font-display); font-size:1.5rem; flex-shrink:0">
            <?php echo strtoupper(mb_substr($client['company_name'], 0, 1)); ?>
        </div>
        <div>
            <h1 class="d-flex align-items-center gap-2 flex-wrap" style="font-size:1.6rem">
                <?php echo h($client['company_name']); ?>
                <?php if ((int)$client['is_active']): ?>
                    <span class="lp-badge paid" style="font-size:.65rem">Active</span>
                <?php else: ?>
                    <span class="lp-badge cancelled" style="font-size:.65rem">Inactive</span>
                <?php endif; ?>
            </h1>
            <div class="lp-subtitle">
                Client since <?php echo date(DATE_FORMAT, strtotime($client['created_at'])); ?>
                <?php if ($client['tax_id']): ?>
                    &middot; Tax ID: <?php echo h($client['tax_id']); ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php if (is_accountant()): ?>
    <div class="d-flex gap-2 flex-wrap">
        <a href="<?php echo APP_URL; ?>/invoice_form.php?client_id=<?php echo $client_id; ?>"
           class="btn btn-lp-primary">
            <i class="bi bi-plus-lg me-1"></i> New Invoice
        </a>
        <a href="<?php echo APP_URL; ?>/client_form.php?id=<?php echo $client_id; ?>"
           class="btn btn-lp-outline">
            <i class="bi bi-pencil me-1"></i> Edit
        </a>
        <a href="<?php echo APP_URL; ?>/clients.php" class="btn btn-lp-outline">
            <i class="bi bi-arrow-left me-1"></i> All Clients
        </a>
    </div>
    <?php endif; ?>
</div>

<!-- ═══ Stat Cards ══════════════════════════════════════════════ -->
<div class="row g-3 mb-4">
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon blue"><i class="bi bi-receipt"></i></div>
                <div>
                    <div class="lp-stat-label">Total Invoiced</div>
                    <div class="lp-stat-value"><?php echo format_currency($stat_total_invoiced); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon emerald"><i class="bi bi-check-circle"></i></div>
                <div>
                    <div class="lp-stat-label">Total Paid</div>
                    <div class="lp-stat-value"><?php echo format_currency($stat_total_paid); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon <?php echo $stat_balance_due > 0 ? 'amber' : 'emerald'; ?>">
                    <i class="bi bi-hourglass-split"></i>
                </div>
                <div>
                    <div class="lp-stat-label">Balance Due</div>
                    <div class="lp-stat-value"><?php echo format_currency($stat_balance_due); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon <?php echo $stat_overdue_count > 0 ? 'red' : 'purple'; ?>">
                    <i class="bi bi-<?php echo $stat_overdue_count > 0 ? 'exclamation-triangle' : 'file-earmark-text'; ?>"></i>
                </div>
                <div>
                    <div class="lp-stat-label"><?php echo $stat_overdue_count > 0 ? 'Overdue' : 'Invoices'; ?></div>
                    <div class="lp-stat-value"><?php echo $stat_overdue_count > 0 ? $stat_overdue_count : $stat_invoice_count; ?></div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row g-4">

    <!-- ═══ Left Column ═════════════════════════════════════════ -->
    <div class="col-lg-8">

        <!-- ── Invoice History ─────────────────────────────────── -->
        <div class="lp-card mb-4">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-receipt me-1"></i> Invoice History</span>
                <span class="text-muted" style="font-size:.78rem"><?php echo $stat_invoice_count; ?> total</span>
            </div>
            <div class="card-body p-0">
                <?php if (empty($invoices)): ?>
                    <div class="lp-empty">
                        <i class="bi bi-receipt d-block"></i>
                        <p>No invoices yet for this client.</p>
                    </div>
                <?php else: ?>
                <div class="lp-table-responsive">
                    <table class="lp-table">
                        <thead>
                            <tr>
                                <th>Invoice #</th>
                                <th>Status</th>
                                <th>Issue Date</th>
                                <th>Due Date</th>
                                <th class="text-end">Total</th>
                                <th class="text-end">Paid</th>
                                <th class="text-end">Balance</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($invoices as $inv): ?>
                            <tr>
                                <td>
                                    <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$inv['id']; ?>"
                                       class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                                        <?php echo h($inv['invoice_number']); ?>
                                    </a>
                                </td>
                                <td>
                                    <?php
                                    /* Compute effective status: mark overdue dynamically */
                                    $eff_status = $inv['status'];
                                    if (in_array($inv['status'], ['sent','draft']) && $inv['due_date'] < date('Y-m-d')) {
                                        $eff_status = 'overdue';
                                    }
                                    ?>
                                    <span class="lp-badge <?php echo h($eff_status); ?>">
                                        <?php echo h(ucfirst($eff_status)); ?>
                                    </span>
                                </td>
                                <td><?php echo date(DATE_FORMAT, strtotime($inv['issue_date'])); ?></td>
                                <td>
                                    <?php
                                    $is_past_due = ($eff_status === 'overdue');
                                    ?>
                                    <span class="<?php echo $is_past_due ? 'text-danger fw-semibold' : ''; ?>">
                                        <?php echo date(DATE_FORMAT, strtotime($inv['due_date'])); ?>
                                    </span>
                                </td>
                                <td class="text-end"><?php echo format_currency((float)$inv['total']); ?></td>
                                <td class="text-end" style="color:var(--lp-emerald)">
                                    <?php echo format_currency((float)$inv['amount_paid']); ?>
                                </td>
                                <td class="text-end fw-semibold">
                                    <?php if ((float)$inv['balance_due'] > 0): ?>
                                        <span style="color:var(--lp-amber)">
                                            <?php echo format_currency((float)$inv['balance_due']); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-muted"><?php echo format_currency(0); ?></span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($stat_invoice_count > 10 && is_accountant()): ?>
                <div class="text-center py-2 border-top" style="border-color:var(--lp-silver) !important">
                    <a href="<?php echo APP_URL; ?>/invoices.php?client_id=<?php echo $client_id; ?>"
                       class="text-decoration-none" style="font-size:.82rem; font-weight:500">
                        View all <?php echo $stat_invoice_count; ?> invoices →
                    </a>
                </div>
                <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- ── Payment History ─────────────────────────────────── -->
        <div class="lp-card mb-4">
            <div class="card-header d-flex align-items-center justify-content-between">
                <span><i class="bi bi-credit-card me-1"></i> Payment History</span>
                <?php if (is_accountant()): ?>
                <a href="<?php echo APP_URL; ?>/payment_form.php?client_id=<?php echo $client_id; ?>"
                   class="btn btn-sm btn-lp-primary" style="font-size:.78rem; padding:.25rem .65rem">
                    <i class="bi bi-plus-lg me-1"></i> Record Payment
                </a>
                <?php endif; ?>
            </div>
            <div class="card-body p-0">
                <?php if (empty($payments)): ?>
                    <div class="lp-empty">
                        <i class="bi bi-credit-card d-block"></i>
                        <p>No payments recorded for this client.</p>
                    </div>
                <?php else: ?>
                <div class="lp-table-responsive">
                    <table class="lp-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Invoice</th>
                                <th>Method</th>
                                <th>Reference</th>
                                <th class="text-end">Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($payments as $pay): ?>
                            <tr>
                                <td><?php echo date(DATE_FORMAT, strtotime($pay['payment_date'])); ?></td>
                                <td>
                                    <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$pay['id']; ?>"
                                       class="text-decoration-none" style="color:var(--lp-blue); font-size:.84rem">
                                        <?php echo h($pay['invoice_number']); ?>
                                    </a>
                                </td>
                                <td>
                                    <span class="badge rounded-pill"
                                          style="background:var(--lp-snow); color:var(--lp-slate); border:1px solid var(--lp-silver); font-size:.72rem; font-weight:500">
                                        <?php echo h(ucfirst(str_replace('_', ' ', $pay['method']))); ?>
                                    </span>
                                </td>
                                <td style="font-size:.84rem; color:var(--lp-slate-mid)">
                                    <?php echo h($pay['reference'] ?: '—'); ?>
                                </td>
                                <td class="text-end fw-semibold" style="color:var(--lp-emerald)">
                                    +<?php echo format_currency((float)$pay['amount']); ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>

    </div><!-- /left col -->

    <!-- ═══ Right Column: Sidebar ═══════════════════════════════ -->
    <div class="col-lg-4">

        <!-- ── Contact Information ─────────────────────────────── -->
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-person-lines-fill me-1"></i> Contact Information
            </div>
            <div class="card-body" style="font-size:.855rem">
                <!-- Contact person -->
                <div class="d-flex align-items-center gap-2 mb-3">
                    <div style="width:36px; height:36px; border-radius:50%;
                                background:var(--lp-blue-soft); color:var(--lp-blue);
                                display:flex; align-items:center; justify-content:center;
                                font-size:.85rem; flex-shrink:0">
                        <i class="bi bi-person"></i>
                    </div>
                    <div>
                        <div class="fw-semibold" style="color:var(--lp-navy)">
                            <?php echo h($client['contact_name']); ?>
                        </div>
                        <small class="text-muted">Primary Contact</small>
                    </div>
                </div>

                <!-- Email -->
                <div class="mb-2 d-flex align-items-center gap-2">
                    <i class="bi bi-envelope text-muted" style="width:18px; text-align:center"></i>
                    <a href="mailto:<?php echo h($client['email']); ?>"
                       class="text-decoration-none" style="color:var(--lp-blue)">
                        <?php echo h($client['email']); ?>
                    </a>
                </div>

                <!-- Phone -->
                <?php if ($client['phone']): ?>
                <div class="mb-2 d-flex align-items-center gap-2">
                    <i class="bi bi-telephone text-muted" style="width:18px; text-align:center"></i>
                    <span><?php echo h($client['phone']); ?></span>
                </div>
                <?php endif; ?>

                <!-- Tax ID -->
                <?php if ($client['tax_id']): ?>
                <div class="mb-0 d-flex align-items-center gap-2">
                    <i class="bi bi-upc-scan text-muted" style="width:18px; text-align:center"></i>
                    <span><?php echo h($client['tax_id']); ?></span>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- ── Address ─────────────────────────────────────────── -->
        <?php
        $has_address = $client['address_line1'] || $client['city'] || $client['state'] || $client['postal_code'];
        if ($has_address):
        ?>
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-geo-alt me-1"></i> Address
            </div>
            <div class="card-body" style="font-size:.855rem; line-height:1.7">
                <?php
                $addr_parts = [];
                if ($client['address_line1']) $addr_parts[] = h($client['address_line1']);
                if ($client['address_line2']) $addr_parts[] = h($client['address_line2']);

                $city_state = [];
                if ($client['city'])        $city_state[] = h($client['city']);
                if ($client['state'])       $city_state[] = h($client['state']);
                if ($client['postal_code']) $city_state[] = h($client['postal_code']);
                if (!empty($city_state))    $addr_parts[] = implode(', ', $city_state);

                if ($client['country'])     $addr_parts[] = h($client['country']);

                echo implode('<br>', $addr_parts);
                ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- ── Portal Access ───────────────────────────────────── -->
        <?php if (is_accountant()): ?>
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-shield-lock me-1"></i> Portal Access
            </div>
            <div class="card-body" style="font-size:.855rem">
                <?php if ($linked_user): ?>
                    <div class="d-flex align-items-center gap-2 mb-2">
                        <span style="width:8px; height:8px; border-radius:50; flex-shrink:0;
                                     background:<?php echo (int)$linked_user['is_active'] ? 'var(--lp-emerald)' : 'var(--lp-slate-light)'; ?>;
                                     display:inline-block; border-radius:50%"></span>
                        <span class="fw-semibold" style="color:var(--lp-navy)">
                            <?php echo h($linked_user['full_name']); ?>
                        </span>
                    </div>
                    <div class="text-muted mb-1">
                        <?php echo h($linked_user['email']); ?>
                    </div>
                    <?php if ($linked_user['last_login_at']): ?>
                    <small class="text-muted">
                        Last login: <?php echo date(DATETIME_FORMAT, strtotime($linked_user['last_login_at'])); ?>
                    </small>
                    <?php else: ?>
                    <small class="text-muted">Never logged in</small>
                    <?php endif; ?>
                <?php else: ?>
                    <div class="text-muted mb-2">
                        <i class="bi bi-info-circle me-1"></i>
                        No portal account linked.
                    </div>
                    <a href="<?php echo APP_URL; ?>/users.php?action=create&client_id=<?php echo $client_id; ?>"
                       class="btn btn-sm btn-lp-outline w-100">
                        <i class="bi bi-person-plus me-1"></i> Create Portal Account
                    </a>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- ── Internal Notes ──────────────────────────────────── -->
        <?php if (is_accountant() && $client['notes']): ?>
        <div class="lp-card mb-4">
            <div class="card-header">
                <i class="bi bi-chat-left-text me-1"></i> Internal Notes
            </div>
            <div class="card-body" style="font-size:.855rem; color:var(--lp-slate-mid); white-space:pre-wrap; line-height:1.6">
<?php echo h($client['notes']); ?>
            </div>
        </div>
        <?php endif; ?>

    </div><!-- /sidebar -->
</div><!-- /row -->

<?php require_once __DIR__ . '/footer.php'; ?>
