<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  client_form.php — Create / Edit Client
 * ═══════════════════════════════════════════════════════════════
 *
 *  Dual-mode form:
 *    - GET  /client_form.php          → blank "Add Client" form
 *    - GET  /client_form.php?id=5     → pre-filled "Edit Client" form
 *    - POST /client_form.php          → validate + INSERT
 *    - POST /client_form.php?id=5     → validate + UPDATE
 *
 *  Security:
 *    - Accountant role required
 *    - CSRF token on every submission
 *    - All input sanitized + validated server-side
 *    - Prepared statements for all DB operations
 *    - Values stick on validation failure (no data loss)
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

/* ══════════════════════════════════════════════════════════════
 *  DETERMINE MODE: Create vs Edit
 * ══════════════════════════════════════════════════════════════ */
$edit_id = isset($_GET['id']) ? (int) $_GET['id'] : 0;
$is_edit = $edit_id > 0;
$errors  = [];

/* Default field values (blank for create) */
$f = [
    'company_name'  => '',
    'contact_name'  => '',
    'email'         => '',
    'phone'         => '',
    'tax_id'        => '',
    'address_line1' => '',
    'address_line2' => '',
    'city'          => '',
    'state'         => '',
    'postal_code'   => '',
    'country'       => 'US',
    'notes'         => '',
    'is_active'     => 1,
];

/* ── If editing, load existing client data ────────────────────── */
if ($is_edit) {
    $client = db_select_one("SELECT * FROM clients WHERE id = ?", ['i', $edit_id]);
    if (!$client) {
        flash('error', 'Client not found.');
        redirect('clients.php');
    }
    /* Populate defaults from database */
    foreach ($f as $key => $val) {
        if (isset($client[$key])) {
            $f[$key] = $client[$key];
        }
    }
}

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Validate & Save
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    /* ── CSRF ──────────────────────────────────────────────── */
    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        $errors[] = 'Invalid security token. Please try again.';
    }

    /* ── Collect & trim all fields ─────────────────────────── */
    $f['company_name']  = trim($_POST['company_name']  ?? '');
    $f['contact_name']  = trim($_POST['contact_name']  ?? '');
    $f['email']         = trim($_POST['email']         ?? '');
    $f['phone']         = trim($_POST['phone']         ?? '');
    $f['tax_id']        = trim($_POST['tax_id']        ?? '');
    $f['address_line1'] = trim($_POST['address_line1'] ?? '');
    $f['address_line2'] = trim($_POST['address_line2'] ?? '');
    $f['city']          = trim($_POST['city']          ?? '');
    $f['state']         = trim($_POST['state']         ?? '');
    $f['postal_code']   = trim($_POST['postal_code']   ?? '');
    $f['country']       = trim($_POST['country']       ?? 'US');
    $f['notes']         = trim($_POST['notes']         ?? '');
    $f['is_active']     = isset($_POST['is_active']) ? 1 : 0;

    /* ── Validation ────────────────────────────────────────── */
    if ($f['company_name'] === '') {
        $errors[] = 'Company name is required.';
    } elseif (mb_strlen($f['company_name']) > 200) {
        $errors[] = 'Company name must be 200 characters or less.';
    }

    if ($f['contact_name'] === '') {
        $errors[] = 'Contact name is required.';
    } elseif (mb_strlen($f['contact_name']) > 120) {
        $errors[] = 'Contact name must be 120 characters or less.';
    }

    if ($f['email'] === '') {
        $errors[] = 'Email address is required.';
    } elseif (!filter_var($f['email'], FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Please enter a valid email address.';
    } elseif (mb_strlen($f['email']) > 255) {
        $errors[] = 'Email address is too long.';
    }

    if ($f['phone'] !== '' && mb_strlen($f['phone']) > 30) {
        $errors[] = 'Phone number must be 30 characters or less.';
    }

    if ($f['tax_id'] !== '' && mb_strlen($f['tax_id']) > 50) {
        $errors[] = 'Tax ID must be 50 characters or less.';
    }

    if ($f['country'] === '') {
        $errors[] = 'Country is required.';
    }

    /* ── Duplicate email check (only among OTHER clients) ──── */
    if (empty($errors)) {
        $dup_sql = "SELECT id FROM clients WHERE email = ?";
        $dup_params = ['s', $f['email']];

        if ($is_edit) {
            $dup_sql .= " AND id != ?";
            $dup_params[0] .= 'i';
            $dup_params[] = $edit_id;
        }

        $dup = db_select_one($dup_sql, $dup_params);
        if ($dup) {
            $errors[] = 'Another client already uses this email address.';
        }
    }

    /* ── Save ──────────────────────────────────────────────── */
    if (empty($errors)) {
        if ($is_edit) {
            db_execute(
                "UPDATE clients SET
                    company_name = ?, contact_name = ?, email = ?, phone = ?,
                    tax_id = ?, address_line1 = ?, address_line2 = ?,
                    city = ?, state = ?, postal_code = ?, country = ?,
                    notes = ?, is_active = ?
                 WHERE id = ?",
                [
                    'ssssssssssssii',
                    $f['company_name'], $f['contact_name'], $f['email'], $f['phone'],
                    $f['tax_id'], $f['address_line1'], $f['address_line2'],
                    $f['city'], $f['state'], $f['postal_code'], $f['country'],
                    $f['notes'], $f['is_active'],
                    $edit_id
                ]
            );
            flash('success', 'Client updated successfully.');
        } else {
            $new_id = db_insert(
                "INSERT INTO clients
                    (company_name, contact_name, email, phone, tax_id,
                     address_line1, address_line2, city, state, postal_code,
                     country, notes, is_active)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                [
                    'ssssssssssssi',
                    $f['company_name'], $f['contact_name'], $f['email'], $f['phone'],
                    $f['tax_id'], $f['address_line1'], $f['address_line2'],
                    $f['city'], $f['state'], $f['postal_code'], $f['country'],
                    $f['notes'], $f['is_active']
                ]
            );
            flash('success', 'Client created successfully.');
            $edit_id = $new_id; // redirect to view
        }

        csrf_regenerate();
        redirect('client_view.php?id=' . $edit_id);
    }
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = $is_edit ? 'Edit Client' : 'Add Client';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1>
            <i class="bi bi-<?php echo $is_edit ? 'pencil-square' : 'person-plus'; ?> me-2" style="font-size:1.4rem"></i>
            <?php echo $is_edit ? 'Edit Client' : 'New Client'; ?>
        </h1>
        <div class="lp-subtitle">
            <?php echo $is_edit
                ? 'Update details for ' . h($f['company_name'])
                : 'Fill in the details below to register a new client.'; ?>
        </div>
    </div>
    <a href="<?php echo APP_URL; ?>/clients.php" class="btn btn-lp-outline">
        <i class="bi bi-arrow-left me-1"></i> Back to Clients
    </a>
</div>

<!-- ═══ Validation Errors ═══════════════════════════════════════ -->
<?php if (!empty($errors)): ?>
<div class="lp-flash error" style="animation:none;">
    <div>
        <i class="bi bi-exclamation-triangle-fill me-1"></i>
        <strong>Please correct the following:</strong>
        <ul class="mb-0 mt-1 ps-3" style="font-weight:400">
            <?php foreach ($errors as $err): ?>
            <li><?php echo h($err); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
</div>
<?php endif; ?>

<!-- ═══ Client Form ═════════════════════════════════════════════ -->
<form method="POST"
      action="client_form.php<?php echo $is_edit ? '?id=' . $edit_id : ''; ?>"
      class="lp-form"
      novalidate>

    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
           value="<?php echo csrf_token(); ?>">

    <div class="row g-4">

        <!-- ═══ Left Column: Core Info ══════════════════════════ -->
        <div class="col-lg-8">

            <!-- Company Information -->
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-building me-1"></i> Company Information
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <!-- Company name -->
                        <div class="col-md-6">
                            <label for="company_name" class="form-label">
                                Company Name <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control" id="company_name" name="company_name"
                                   value="<?php echo h($f['company_name']); ?>"
                                   placeholder="Acme Industries" required maxlength="200">
                        </div>

                        <!-- Contact name -->
                        <div class="col-md-6">
                            <label for="contact_name" class="form-label">
                                Contact Person <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control" id="contact_name" name="contact_name"
                                   value="<?php echo h($f['contact_name']); ?>"
                                   placeholder="John Doe" required maxlength="120">
                        </div>

                        <!-- Email -->
                        <div class="col-md-6">
                            <label for="email" class="form-label">
                                Email Address <span class="text-danger">*</span>
                            </label>
                            <input type="email" class="form-control" id="email" name="email"
                                   value="<?php echo h($f['email']); ?>"
                                   placeholder="john@acme.com" required maxlength="255">
                        </div>

                        <!-- Phone -->
                        <div class="col-md-3">
                            <label for="phone" class="form-label">Phone</label>
                            <input type="tel" class="form-control" id="phone" name="phone"
                                   value="<?php echo h($f['phone']); ?>"
                                   placeholder="+1 555-0100" maxlength="30">
                        </div>

                        <!-- Tax ID -->
                        <div class="col-md-3">
                            <label for="tax_id" class="form-label">Tax ID / EIN</label>
                            <input type="text" class="form-control" id="tax_id" name="tax_id"
                                   value="<?php echo h($f['tax_id']); ?>"
                                   placeholder="XX-XXXXXXX" maxlength="50">
                        </div>
                    </div>
                </div>
            </div>

            <!-- Address -->
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-geo-alt me-1"></i> Address
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <!-- Address Line 1 -->
                        <div class="col-12">
                            <label for="address_line1" class="form-label">Street Address</label>
                            <input type="text" class="form-control" id="address_line1" name="address_line1"
                                   value="<?php echo h($f['address_line1']); ?>"
                                   placeholder="742 Evergreen Terrace" maxlength="255">
                        </div>

                        <!-- Address Line 2 -->
                        <div class="col-12">
                            <label for="address_line2" class="form-label">Apt / Suite / Floor</label>
                            <input type="text" class="form-control" id="address_line2" name="address_line2"
                                   value="<?php echo h($f['address_line2']); ?>"
                                   placeholder="Suite 200" maxlength="255">
                        </div>

                        <!-- City -->
                        <div class="col-md-4">
                            <label for="city" class="form-label">City</label>
                            <input type="text" class="form-control" id="city" name="city"
                                   value="<?php echo h($f['city']); ?>"
                                   placeholder="Springfield" maxlength="100">
                        </div>

                        <!-- State -->
                        <div class="col-md-3">
                            <label for="state" class="form-label">State / Province</label>
                            <input type="text" class="form-control" id="state" name="state"
                                   value="<?php echo h($f['state']); ?>"
                                   placeholder="IL" maxlength="100">
                        </div>

                        <!-- Postal code -->
                        <div class="col-md-2">
                            <label for="postal_code" class="form-label">Postal Code</label>
                            <input type="text" class="form-control" id="postal_code" name="postal_code"
                                   value="<?php echo h($f['postal_code']); ?>"
                                   placeholder="62704" maxlength="20">
                        </div>

                        <!-- Country -->
                        <div class="col-md-3">
                            <label for="country" class="form-label">
                                Country <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" id="country" name="country" required>
                                <option value="">Select…</option>
                                <?php
                                $countries = [
                                    'US' => 'United States',
                                    'CA' => 'Canada',
                                    'GB' => 'United Kingdom',
                                    'AU' => 'Australia',
                                    'DE' => 'Germany',
                                    'FR' => 'France',
                                    'NL' => 'Netherlands',
                                    'CH' => 'Switzerland',
                                    'SE' => 'Sweden',
                                    'NO' => 'Norway',
                                    'DK' => 'Denmark',
                                    'IE' => 'Ireland',
                                    'NZ' => 'New Zealand',
                                    'SG' => 'Singapore',
                                    'JP' => 'Japan',
                                    'IN' => 'India',
                                    'BR' => 'Brazil',
                                    'MX' => 'Mexico',
                                    'ZA' => 'South Africa',
                                    'AE' => 'United Arab Emirates',
                                    'TR' => 'Turkey',
                                    'OTHER' => 'Other',
                                ];
                                foreach ($countries as $code => $name):
                                ?>
                                <option value="<?php echo h($code); ?>"
                                    <?php echo $f['country'] === $code ? 'selected' : ''; ?>>
                                    <?php echo h($name); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Notes -->
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-chat-left-text me-1"></i> Notes
                </div>
                <div class="card-body">
                    <textarea class="form-control" id="notes" name="notes" rows="4"
                              placeholder="Internal notes about this client (not visible to the client)…"
                              style="resize:vertical"><?php echo h($f['notes']); ?></textarea>
                    <small class="text-muted mt-1 d-block">
                        Optional. These notes are only visible to accountants.
                    </small>
                </div>
            </div>
        </div>

        <!-- ═══ Right Column: Sidebar ═══════════════════════════ -->
        <div class="col-lg-4">

            <!-- Status Card -->
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-toggle-on me-1"></i> Status
                </div>
                <div class="card-body">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" role="switch"
                               id="is_active" name="is_active" value="1"
                               <?php echo $f['is_active'] ? 'checked' : ''; ?>
                               style="width:2.8em; height:1.4em; cursor:pointer">
                        <label class="form-check-label ms-2" for="is_active"
                               style="font-weight:500; cursor:pointer">
                            Client is active
                        </label>
                    </div>
                    <small class="text-muted d-block mt-2">
                        Inactive clients are hidden from dropdowns but their
                        data is preserved.
                    </small>
                </div>
            </div>

            <!-- Quick Reference (edit mode only) -->
            <?php if ($is_edit): ?>
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-info-circle me-1"></i> Quick Info
                </div>
                <div class="card-body" style="font-size:.84rem">
                    <div class="d-flex justify-content-between mb-2">
                        <span class="text-muted">Client ID</span>
                        <span class="fw-semibold">#<?php echo $edit_id; ?></span>
                    </div>
                    <div class="d-flex justify-content-between mb-2">
                        <span class="text-muted">Created</span>
                        <span><?php echo date(DATE_FORMAT, strtotime($client['created_at'])); ?></span>
                    </div>
                    <div class="d-flex justify-content-between mb-2">
                        <span class="text-muted">Last Updated</span>
                        <span><?php echo date(DATE_FORMAT, strtotime($client['updated_at'])); ?></span>
                    </div>
                    <?php
                    $ci_count = (int) db_scalar(
                        "SELECT COUNT(*) FROM invoices WHERE client_id = ?",
                        ['i', $edit_id]
                    );
                    $ci_balance = (float) db_scalar(
                        "SELECT COALESCE(SUM(balance_due), 0) FROM invoices
                         WHERE client_id = ? AND status NOT IN ('paid','cancelled')",
                        ['i', $edit_id]
                    );
                    ?>
                    <hr class="my-2" style="border-color:var(--lp-silver)">
                    <div class="d-flex justify-content-between mb-2">
                        <span class="text-muted">Invoices</span>
                        <span class="fw-semibold"><?php echo $ci_count; ?></span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <span class="text-muted">Balance Due</span>
                        <span class="fw-semibold"
                              style="color:<?php echo $ci_balance > 0 ? 'var(--lp-amber)' : 'var(--lp-emerald)'; ?>">
                            <?php echo format_currency($ci_balance); ?>
                        </span>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Actions -->
            <div class="lp-card">
                <div class="card-body d-grid gap-2">
                    <button type="submit" class="btn btn-lp-primary">
                        <i class="bi bi-check-lg me-1"></i>
                        <?php echo $is_edit ? 'Update Client' : 'Create Client'; ?>
                    </button>
                    <a href="<?php echo APP_URL; ?>/clients.php" class="btn btn-lp-outline">
                        Cancel
                    </a>
                    <?php if ($is_edit): ?>
                    <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo $edit_id; ?>"
                       class="btn btn-lp-outline">
                        <i class="bi bi-eye me-1"></i> View Profile
                    </a>
                    <?php endif; ?>
                </div>
            </div>

        </div><!-- /sidebar -->
    </div><!-- /row -->
</form>

<?php require_once __DIR__ . '/footer.php'; ?>
